/* 
   I2C SLAVE - LCD display controlled by 3 wires using 74HC595 serial shift register
   ---------------------------------------------------------------------------------

   Software version: 0.04 (compatible with hardware version 0.02)
   Author: M.Post
   Last modified: 2 Jan 2013
   

	
   DONE:
     
 
   TODO:
   
     - move lcd messages to PROGMEM
   
     - in setup_t85, see if we need to move LCD_init *before* calling menu
	 
	 - in setup_t85, SDA and SCL should be HIGH before menu is shown
   
     - connect PB5(RST) to LCD Backlight and accept turn on/off instructions over I2C
	    for this to work, PB5 needs to be made available as I/O by setting the HIGH
		fuse byte to 0x5F. This command disables the RESET function of the mcu which will
		also prevent reprogramming the mcu using a standard 5V programmer.
		See discussion here:  http://arduino.cc/forum/index.php/topic,87517.0.html

		In order to ever reprogram the mcu you will need to enable the RST function by
 		using a 12V fuse byte resetter like the one from here: 
		http://www.rickety.us/2010/03/arduino-avr-high-voltage-serial-programmer/
		
		NOTE: Revision 0.02 of the board should have a track from pin 1 (PB5) to a pad labeled (HV_PRG).
		      For normal production and usage this pad should be left unsoldered.
		
	 
     - eeprom value for I2C id may need to be set explicitly for brand new mcu's.
        I wrote a simple function that checks if the value is 255 but I'm not sure
        if this is adequate for new chips. If your display hangs at startup it may 
        just be that you need to write an eeprom value first:
           - change code to write a default I2C value 
           - program the chip
           - change the code back to only write a code if it's 255
           - program the chip

     - add button(s) to send I2C commands?

     special instructions:
       - clear one row
       - turn on backlight  (PB5 cannot do PWM!)
            pins that can do pwm: 
             PB0 output pin for PWM mode timer function
             PB1 output pin for PWM mode timer function 
             PB3 inverted output pin for PWM mode timer function
             PB4 output pin for PWM mode timer function

	PWM Code example:
	http://www.nongnu.org/avr-libc/user-manual/group__demo__project.html


                             ATtiny85
               
                             +--- ---+
       backlight  (RST) PB5 -|1 |_| 8|- VCC
                    EN  PB3 -|2     7|- PB2 SCL    (SCK)          
    setup switch    CLK PB4 -|3     6|- PB1 SDA    (MISO)         
                        GND -|4     5|- PB0 Dout   (MOSI)
                             +-------+

     // clock is defined on gcc command line using 'DF_CPU=8000000UL'
     // you may therefore need to program the lower fuse byte to 0xE1 (disable clkdiv8)

      LCD Connector: GND VCC EN DOUT CLK

      LCD Pinout
      ----------
      15 Led Power Anode (driven by BC338 through PB5)
      16 Led Power Cathode (to GND)
      1 Vss (GND)
      2 Vdd (+5V)
      3 VO
      4 RS ----------- to pin  4 on 74HC595 (Q4)
      5 R/W
      6 E ------------ to 'EN' pin on t85
      7 DB0
      8 DB1
      9 DB2
      10 DB3
      11 DB4 --------- to pin 15 on 74HC595 (Q0)
      12 DB5 --------- to pin  1 on 74HC595 (Q1)
      13 DB6 --------- to pin  2 on 74HC595 (Q2)
      14 DB7 --------- to pin  3 on 74HC595 (Q3)


      74HC595
      -------
      pinout as per above, plus:
       - pin 11 (SH_CP) and pin 12 (ST_CP) are tied together
       - above pins to to CLK pin on t85
       - pin 14 (DS) to Dout pin on t85
       - pin 8 and 13 to GND
       - pin 16 and 10 to +5V


      Converter board pinout
      ----------------------
      
          +----------------------------------------+
          |                                        |
          |     +---------------------+            |
          |     |                     |    o +5V   |
          |     |   LCD Module        |    o SCL   |
          |  p  |                     |    o SDA   |
          |     +---------------------+    o GND   |
          |      r1 74HC595 t85  btn  isp          |
          |                                        |
          +----------------------------------------+

     
 */



#include <avr/io.h>
#include <util/delay.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>
#include <avr/eeprom.h>

// pins to be used on ATtiny85
#define STR        PB3  // Define Strobe/Enable pin  (doubles as setup switch pin)
#define Dout       PB0  // Define Data-Out pin 
#define CLK        PB4  // Define Clock pin  
#define SCL        PB2  // Define I2C Clock pin  
#define SDA        PB1  // Define I2C Data pin  
#define BACKLIGHT  PB5  // Define LED Backlight pin  (not yet implemented?)

#define HIGH   1
#define LOW    0
#define INPUT  1 
#define OUTPUT 0

int device_id;                     // this slave's device id (fetch from eeprom)
int id_match;                      // high when master addresses us
int start=0;                       // repeated start indicator
unsigned int i2c_timeout=65535;    // 4185000=5sec   // when to give up listening for more bits  (837,000 is about 1 second @ 8MHz)

// -- begin I/O functions --
// -- begin I/O functions --
// -- begin I/O functions --

void digitalWrite(int pin, int state)
{
  if (state == HIGH)
  { 
    (PORTB |= (1 << pin)); 
  } else {
    (PORTB &= ~(1 << pin));
  }
} 

int digitalRead(int pin)
{ 
  if (bit_is_set(PINB,pin))
  {
    return 1; 
  } else { 
    return 0; 
  }
} 

void pinMode(int pin,int state)
{ 
  if (state == OUTPUT)
  {
    (DDRB |= (1 << pin));  // set pin to output
  } else { 
    (DDRB &= ~(1 << pin)); // set pin to input
  }
} 

// -- end I/O functions --


// -- begin LCD functions --
// -- begin LCD functions --
// -- begin LCD functions --

void Write_LCD_Nibble(unsigned short N, int RS)
{

  digitalWrite(STR,0);

  // Write RS 
  digitalWrite(CLK,0);
  digitalWrite(Dout,RS);  // Data_Pin
  digitalWrite(CLK,1);
  digitalWrite(CLK,0);


  // Shift in 4 bits
  int t = 0;
  int Flag = 0;
  int Mask = 8;

  for (t=0; t<4; t++)
  {
    Flag = N & Mask;
    if(Flag==0)
    {
      digitalWrite(Dout,0);  // Data_Pin     
    } else {
      digitalWrite(Dout,1);  // Data_Pin
    }
 
    digitalWrite(CLK,1);
    digitalWrite(CLK,0);

    Mask = Mask >> 1;
  }

  // One more clock because SC and ST clks are tied
  digitalWrite(CLK,1);
  digitalWrite(CLK,0);
  digitalWrite(Dout,0);  // Data_Pin
  digitalWrite(STR,1);   // Enable_Pin  (to LCD)
  digitalWrite(STR,0);   // Enable_Pin  (to LCD)

}

void Write_LCD_Data(unsigned short D)
{
  int RS = 1; // It is Data, not command
  int Low_Nibble = D & 15;
  int High_Nibble = D/16;
  Write_LCD_Nibble(High_Nibble,RS);
  Write_LCD_Nibble(Low_Nibble,RS);
}

void Write_LCD_Cmd(unsigned short C)
{

  int RS = 0; // It is command, not data
  int Low_Nibble = C & 15;
  int High_Nibble = C/16;
  Write_LCD_Nibble(High_Nibble,RS);
  Write_LCD_Nibble(Low_Nibble,RS);
}

void initialise_LCD() 
{

  pinMode(CLK,0);
  pinMode(STR,0);
  pinMode(Dout,0);

  _delay_ms(100); 

  Write_LCD_Cmd(0x03); // function set
  _delay_ms(50);

  Write_LCD_Cmd(0x03);
  _delay_ms(50);

  Write_LCD_Cmd(0x03);
  _delay_ms(50);

  Write_LCD_Cmd(0x02);
  _delay_ms(50);

  Write_LCD_Cmd(0x2C); // Display ON, No cursors
  _delay_ms(50);

  Write_LCD_Cmd(0x06); // Entry mode- Auto-increment, No Display shifting
  _delay_ms(50);

  Write_LCD_Cmd(0x0E);
  _delay_ms(50);

  Write_LCD_Cmd(0x01);
  _delay_ms(50);

  Write_LCD_Cmd(0x80);
  _delay_ms(50);

}

void Position_LCD(unsigned short x, unsigned short y)
{
  int temp = 127 + y;

  if (x == 2) 
  {
    temp = temp + 64;
  }

  Write_LCD_Cmd(temp);
}
 
void Clear_LCD()
{
   Write_LCD_Cmd(0x01);  // Clear LCD
   _delay_ms(5);

}

void Write_LCD_Text(char *StrData)
{

  int p = 0;
  int q = strlen(StrData);
  int temp = 0;

  for (p = 0; p < q; p++)
  {
    temp = StrData[p];
    Write_LCD_Data(temp);
  }
}

void Write_LCD_Number(char number)
{
  // writes a number's digits on the display

  int digits;

  if (number == 0)
  {
    digits = 0;
    int array[0];
    array[0]=0;
    Write_LCD_Data(array[0]+48);

  } else {
 
    // determine the number of digits 
    int digits = (int)(log(number)/log(10))+1;

    // split up the number to an array
    int i = digits - 1;
    int array[digits];
    while (number > 0)
    {
      array[i--] = number % 10;
      number /= 10;  
    }

    // display array on LCD
    for(i = 0; i <= digits-1; i++)
    {
      Write_LCD_Data(array[i]+48);
    }

  }

}

// -- end LCD functions --


// -- begin i2c functions --
// -- begin i2c functions --
// -- begin i2c functions --

void i2cpin(int line, int state)
{

  if (state == 0)
  {
    pinMode(line,OUTPUT);
    digitalWrite(line,LOW); 
  } else {
    pinMode(line,INPUT);
  }

}

void clock_stretch_on()
{
  // enable clock stretching by slave
  
  i2cpin(SCL,LOW);
  
}

void clock_stretch_off()
{
  // disable clock stretching by slave
  i2cpin(SCL,HIGH);
  
}

void wait_for_start()
{
  // exits loop whenever a start is detected and clock goes low
 
  while (1)
  {
    
    while (bit_is_clear(PINB,SDA)) {} // wait for SDA to go high 
    while (bit_is_set(PINB,SDA)) {} // wait for SDA to go low
    if (bit_is_set(PINB,SCL))
    {
      // i2c start sequence detected, exit loop    
      break;
    } 

  }

  while (bit_is_set(PINB,SCL)) {} // exit when SCL goes low

}

unsigned char i2c_rxbyte(int type)
{
  // receives one byte and issues ACK upon completion.
  
  // type is either 0: detect i2c_id or 1:data-bytes
  
  unsigned char rxbyte=0;  // will contain the full byte received
  int bitcount=8;
  unsigned int counter;
  int err=0;

  // stop any clock stretching: we now release SCL 
  (DDRB &= ~(1 << SCL)); // set pin to input
  
  
  // this loops goes 8x, once for every Byte-bit
  // the ACK bit is handled below this loop.
  while (bitcount > 0)
  {  
    
    counter=0;

    // wait for SCL to go high 
    while (counter < i2c_timeout) 
    {
      if (bit_is_set(PINB,SCL))
      {
        if (bit_is_set(PINB,SDA))
        {
          // -- detected bit '1' --          
          rxbyte |= 1 << 0 ; // sets bit 0 of rxbyte
		  
        } else {

          // -- detected bit '0' --
		  // no bit-setting required
		 
        }

        break;       
      }  
      counter++;
    }  
    // scl is either high or low
    // sda is either high or low
	
    
    if (counter >= i2c_timeout)
    {
      // timeout reached! (SCL never went high) 
      // stop listening for further bits/bytes
      err=1;
	  
      break;
	  
    } else {
	
	  // still here? means we received a valid SCL-high
	  
	}
	// scl is high 
    // sda is either high or low

	
	counter=0; // moved counter a little earlier to suppress the small blip on SDA
	
	
	// for the 8th bit: bitcount=1
	
    // shift rxbyte and continue listening
	// we only do this for the first 7 bits, the last bit can stay in place
    if (bitcount > 1)
    {
	
      rxbyte = rxbyte << 1; // shift all bits in rxbyte one position to the left
    
	}
  
    // scl is high 
    // sda is either high or low 

		
    // wait for SCL to go low again	
    while (counter < i2c_timeout)
    {
	
	 
	  
      if (bit_is_clear(PINB,SCL))
      {
        // SCL is now low		
		if (bitcount == 1)
        {
	      // reached 8th pass of receiving individual bits
	      // on this last pass we bring/keep SDA low to show a nice
		  // flat line so the master can detect our ACK.
		  
          (DDRB |= (1 << SDA));     // set pin to output
          (PORTB &= ~(1 << SDA));   // write a '0'
	    }
        break; 
      }
	  counter ++; 
    }

	// scl is high 
    // sda is either high or low (always low after receiving the 8th bit)

	
	if (counter >= i2c_timeout)
    {
      // timeout reached! (SCL never went low) 
      // stop listening for further bits/bytes
      
	  err=2;
      break;
	  
    } else {
	
	  // still here? means we received a valid SCL-low
	  
	}
	
	// scl is low
    // sda is either high or low (always low after receiving the 8th bit)

	
	
	// We have now detected and set a bit for rxbyte. This could
	// be any bit in the 8-bit Byte. If this run processed the 8th
	// bit SDA will now be LOW. We deal with the ACK after this loop.
	
    bitcount--;

  }
  
  // end of receiving Byte-bits
  
  // check for errors, and possibly issue ACK
  
  
  if (err==0)
  {
    
	// scl is low 
	// sda is low (forced low after receiving the 8th bit)

	// we have now received 8x Byte-bits and SDA will be LOW to indicate an ACK.

	// wait for SCL to go high so the master can detect our ACK.
	counter=0;
	while (counter < i2c_timeout) 
	{
		if (bit_is_set(PINB,SCL))
		{
		  // SCL high detected, master should now detect our SDA-low ACK
		  break;       
		}  
		counter++;
	}  
	
	// scl is either high or low
	// sda is either high or low

	if (counter >= i2c_timeout)
	{
	  // timeout reached! (SCL never went high) 
	  // stop listening for further bits/bytes
	 
	  err=3;
	  
	} else {

	  // still here? means we received a valid SCL-high
	  
	}
	// scl is high 
	// sda is low (forced low after receiving the 8th bit)

	
	if (err==0)
	{
	

		// now that the master has sent the ACK clock pulse HIGH on SCL we will
		// wait for the master to bring SCL low again.

		// wait for SCL to go low again
		counter=0;
		while ((counter < i2c_timeout) && (err == 0))
		{
		  if (bit_is_clear(PINB,SCL))
		  {
			// SCL is now low		
			break; 
		  }
		  counter ++; 
		}

		if (counter >= i2c_timeout)
		{
		  // timeout reached! (SCL never went low) 
		  // stop listening for further bits/bytes
		  
		  err=4;
		  
		} else {

		  // still here? means we received a valid SCL-low
		  
		}

		// scl is low
		// sda is low (forced low after receiving the 8th bit)

		// the master has now stopped looking for an ACK bit from the slave
		// we can now safely release SDA and return the received rxbyte.

	}

  }
  
  // enable clock stretching on SCL
  (DDRB |= (1 << SCL));     // set pin to output
  (PORTB &= ~(1 << SCL));   // write a '0'
  
  
  // release SDA
  (DDRB &= ~(1 << SDA)); // set pin to input
	
  if (err > 0)
  {
    
	rxbyte=0;
	id_match=0;
	
  }
  
  if (type>0)
  {
    // show incoming data characters
    //Write_LCD_Data('1');
  }
  	
  return rxbyte;
}


void wait_for_devmatch()
{
  id_match=0;
  unsigned char rxbyte;
  int dev_rx_id = device_id;
  dev_rx_id <<= 1;                // shift byte  

  while (1)
  {
    // stop any clock stretching: we now release SCL 
    (DDRB &= ~(1 << SCL)); // set pin to input
	
    wait_for_start();            // loop until we detect an i2c start sequence
    rxbyte = i2c_rxbyte(0);      // then listen for a byte being transmitted
	// clock stretching always enabled after i2c_rxbyte
	
    if (rxbyte == dev_rx_id)
    {	
	  id_match=1;
      break;
    }

  } 
  
}

// -- end I2C functions --


// -- begin on-board EEPROM and menu functions --
// -- begin on-board EEPROM and menu functions --
// -- begin on-board EEPROM and menu functions --

void setup_menu()
{
  // CLK is the switch pin

  
  int new_dev_id=device_id;
  int counter=0;
  int setup_exit=0;
  char dev_id[]="Device ID: ";

  Clear_LCD();
  Position_LCD(1,1);
  Write_LCD_Text(dev_id);

  while (1)
  {

    // display current value of the device id

    Position_LCD(1,12);
    Write_LCD_Number(new_dev_id);

    pinMode(CLK,1);    // change pinmode so we can listen for button presses
    // pinMode(STR,1);
    // pinMode(Dout,1);


    while (bit_is_clear(PINB, CLK)) {}  // wait until we receive a button press (switch pin)
    _delay_ms(50);

    // see if we hold the button down for a long time to indicate an exit
    counter=0; 
    setup_exit=0;
    while (bit_is_set(PINB, CLK))   // set switch pin here
    {
      _delay_ms(1);
      if (counter >= 3000)
      {
	    // long button press detected, wait until button is released and then break out this loop
        setup_exit=1;
        while (bit_is_set(PINB, CLK)) {}  // wait until button is released (switch pin)
        break;
      }
	  
	  // increment the counter to detect how long the button is being pressed
	  counter++;	  
    } 
	
	// button is now released
	
    pinMode(CLK,0);  // set pin to output to enable LCD writing

    if (setup_exit==0)
    {
	  // button was pressed for a shor amount of time
	  // just update the counter on the LCD
	  
	  
	  /*
   skip reserved i2c id (7-bit) addresses:
   
	0         (interferes with read and write)
	1         CBUS address	
	2         reserved for different bus format
	3         reserved for future purposes
	4-7       Hs-mode master code
	120-123   10-bit slave addressing
	124-127   reserved for future purposes
	
  */
	  
	  
      new_dev_id++;
      if (new_dev_id > 120)
      { 
        new_dev_id=8;
        Clear_LCD();
        Position_LCD(1,1);
        Write_LCD_Text(dev_id);
      }
    } else {

	  // button was pressed for a long amount of time
      // write new device id to eeprom
      eeprom_write_byte(0,new_dev_id);
      device_id=new_dev_id;

      initialise_LCD();
       
      char updated[]="* UPDATED * ";
      Position_LCD(1,1);
      Write_LCD_Text(dev_id);
      Position_LCD(1,12);
      Write_LCD_Number(new_dev_id);
      Position_LCD(2,1);
      Write_LCD_Text(updated);

      _delay_ms(1000); 
      _delay_ms(1000); 
 
      break;

    }
  
  }

}

void enter_setup_check()
{
  // check if Switch is pressed down on startup for longer than 1 second
  pinMode(CLK,1);   // set CLK to input  (switch pin)

  if (bit_is_set(PINB, CLK)) 
  {
    // push button is pressed
   
    while (bit_is_set(PINB,CLK))  { };   // wait until button is released

    pinMode(CLK,0);
    pinMode(STR,0);
    pinMode(Dout,0);

    initialise_LCD();

    char setup_welcome[]="Entering setup..";
    Write_LCD_Text(setup_welcome);

    _delay_ms(1000);
    _delay_ms(1000);
    _delay_ms(1000);

    setup_menu();

  }

}

void get_or_set_i2c_device_id()
{
  // retrieve device id from on-system eeprom
  device_id=eeprom_read_byte(0);
  _delay_ms(10);

 
  if (device_id == 255)
  {
    device_id=58;                 // initial value if eeprom is empty
    eeprom_write_byte(0,device_id);
    _delay_ms(10);
  }
}

// -- end on-board EEPROM and menu functions --

void setup_t85()
{

  // initial setup
  
  // Using the backlight requires the RESET function to be disabled.
  // check high fuse byte for the ATtiny85 to find out how.
  // set the pin controlling the backlight as output
  //pinMode(BACKLIGHT,OUTPUT);
  //digitalWrite(BACKLIGHT,LOW);
  
  // load I2C device ID
  get_or_set_i2c_device_id();

  // display setup menu if button is pressed at boot-up
  enter_setup_check();

  // display the LCD module
  initialise_LCD();

  // set the state of the I2C communication lines
  i2cpin(SDA,HIGH);  
  i2cpin(SCL,HIGH);  

}

int main()
{

  // Initialise LCD module, get i2c address and enter setup menu if required.
  setup_t85();

  int rxbyte=0;

  char Message1[] = "I2C LCD v0.04";
  char Message2[] = "Device ID: ";
 
  Position_LCD(1,1);
  Write_LCD_Text(Message1);

  Position_LCD(2,1);
  Write_LCD_Text(Message2);

  Write_LCD_Number(device_id);
  
  _delay_ms(1000);

  Clear_LCD();

  int LCD_Row, LCD_Col; 


  while (1)
  {
    
    int chars_rx=0;
	
    wait_for_devmatch(); // endless loop to wait for i2c_id, then sets id_match=1
    // clock stretching is now enabled
	
   
    // listen for incoming data..
    while (id_match==1)
    { 
	
      // listen for data packets 
      rxbyte = i2c_rxbyte(1);
	  // clock stretching always enabled after i2c_rxbyte 
	   
      if (rxbyte == 0)
      {
	  
        // no more valid data received, exit this loop
        break;

      } else {

        // valid byte received

        
        // only send valid ASCII characters (32-126) to the LCD
        if ((rxbyte >= 32) && (rxbyte <= 126))
        {
          chars_rx++;
          // regular character
          Write_LCD_Data(rxbyte);
		 
		  
		 
        } else {

          // special character(s)

          // clear screen
          if (rxbyte == 12)
          { 

            Clear_LCD(); 
            
          }

		  
          // positioning of the cursor on the LCD
          if (rxbyte == 10)
          {
            
		  
            // first byte is LCD row (1-2)
            LCD_Row=i2c_rxbyte(1);   // Read data and send acknowledge
            // clock stretching always enabled after i2c_rxbyte 
		  
            // second byte is LCD column (1-16)
            LCD_Col=i2c_rxbyte(1);   // Read data and send acknowledge
            // clock stretching always enabled after i2c_rxbyte           
		  
		  
            // position the cursor
            Position_LCD(LCD_Row,LCD_Col);
            
          }
		  
		  // LCD Backlight control (not implemented yet)
          if (rxbyte == 14)
          {
		    // turn on LCD backlight			
			digitalWrite(BACKLIGHT,HIGH);
			
		  }
		  if (rxbyte == 15)
          {
		    // turn off LCD backlight			
			digitalWrite(BACKLIGHT,LOW);
			
		  }
		  
        } // end if dissecting what kind of data byte we received

        // received byte has now been processed
		
      } // end-if when receiving one byte
 
      
     


    } // while-loop for receiving bytes addressed to us


  } // main listening loop

  return 0;
}
