/* 
   DS1820 temperature sensor
   -------------------------


 
                     ATtiny85
               
                      +--- ---+
    (RST)        PB5 -|1 |_| 8|- VCC
             SDA PB3 -|2     7|- PB2        (SCK)          
             SCL PB4 -|3     6|- PB1        (MISO)         
                 GND -|4     5|- PB0        (MOSI)
                      +-------+          

   clock is defined on gcc command line using 'DF_CPU=8000000UL'



   Pololu programming cable pinout (top view)

                          +---+
    ----------------  gnd |o o| rst
    ribbon cable     mosi |o o| sck
    -----red--------  vdd |o o| miso
                          +---+



   something on progmem:
   http://www.avrfreaks.net/index.php?name=PNphpBB2&file=viewtopic&t=38003



 */


#include <avr/io.h>
#include <util/delay.h>
#include <string.h>
#include <math.h>
#include <avr/pgmspace.h>
 
// pins to be used on ATtiny85
#define SDA       PB3    // Define Data pin  
#define SCL       PB4    // Define Clock pin  

#define THERM_PORT   PORTB
#define THERM_DDR    DDRB
#define THERM_PIN    PINB
#define THERM_DQ     PB0  // Define DS1820 DQ pin

#define THERM_INPUT_MODE()    THERM_DDR&=~(1<<THERM_DQ)
#define THERM_OUTPUT_MODE()   THERM_DDR|=(1<<THERM_DQ)
#define THERM_LOW()           THERM_PORT&=~(1<<THERM_DQ)
#define THERM_HIGH()          THERM_PORT|=(1<<THERM_DQ)

#define HIGH   1
#define LOW    0
#define INPUT  1 
#define OUTPUT 0

#define THERM_CMD_CONVERTTEMP    0x44
#define THERM_CMD_RSCRATCHPAD    0xbe
#define THERM_CMD_WSCRATCHPAD    0x4e
#define THERM_CMD_CPYSCRATCHPAD  0x48
#define THERM_CMD_RECEEPROM      0xb8
#define THERM_CMD_RPWRSUPPLY     0xb4
#define THERM_CMD_SEARCHROM      0xf0
#define THERM_CMD_READROM        0x33
#define THERM_CMD_MATCHROM       0x55
#define THERM_CMD_SKIPROM        0xcc
#define THERM_CMD_ALARMSEARCH    0xec

#define THERM_DECIMAL_STEPS_12BIT   625    // 0.0625

#define PRESENCE_ERR 0xFF
#define DATA_ERR     0xFF
#define LAST_DEVICE  0x00
#define SEARCH_FIRST 0xFF

#define CLEAR(s) memset(&(s), 0, sizeof(s))     // to clear an array


int      got_ack;
uint8_t  romcodes[20][9];  // store romcode of 20 sensors max
uint8_t  scratchpad[9];    // stores a scratchpad

// write constants to EEPROM instead of RAM to save space
prog_char lcdmsg00[] PROGMEM = {"Short Circuit!"};
prog_char lcdmsg01[] PROGMEM = {"No sensors found"};
prog_char lcdmsg02[] PROGMEM = {"Bus Error!"};
prog_char lcdmsg03[] PROGMEM = {"Device "};
prog_char lcdmsg04[] PROGMEM = {"Devices: "};
prog_char lcdmsg05[] PROGMEM = {"Temp: "};
prog_char lcdmsg06[] PROGMEM = {""};

PROGMEM const char *lcdmsg[] = 	  
{
  lcdmsg00,
  lcdmsg01,
  lcdmsg02,
  lcdmsg03,
  lcdmsg04,
  lcdmsg05,
  lcdmsg06,
};

char buffer[16];



void pinMode(int pin,int state)
{ 
  if (state == OUTPUT)
  {
    (DDRB |= (1 << pin)); 
  } else { 
    (DDRB &= ~(1 << pin)); 
  }
} 

void digitalWrite(int pin, int state)
{
  if (state == HIGH)
  { 
    (PORTB |= (1 << pin)); 
  } else {
    (PORTB &= ~(1 << pin));
  }
} 


int digitalRead(int pin)
{ 
  if (bit_is_set(PINB,pin))
  {
    return 1; 
  } else { 
    return 0; 
  }
} 

void i2cpin(int line, int state)
{

  if (state == 0)
  {
    pinMode(line,OUTPUT);
    digitalWrite(line,LOW); 

  } else {
    pinMode(line,INPUT);

  }

}

void i2c_delay()
{
  // default delay in between I2C commands
  _delay_ms(10);   // seems to hover around 2-3ms
}

void i2c_debug_delay()
{
  
  //_delay_us(280);  // breaks below 280us
  _delay_us(1);

}


void i2c_start()
{
  // SDA goes from high to low while SCL is high

  i2cpin(SDA,HIGH); 
  _delay_us(10);    // give bus some time to bring SDA high
  i2cpin(SCL,HIGH);

  _delay_us(10);    

  i2cpin(SDA,LOW);
  _delay_us(10);    

}

void i2c_stop()
{
  // SDA goes from low to high while SCL is high

  i2cpin(SDA,LOW);
  _delay_us(20);    // give bus some time to bring SDA high
  i2cpin(SCL,LOW); 

  _delay_ms(6);

  i2cpin(SCL,HIGH);
  _delay_ms(6);

  i2cpin(SDA,HIGH);
  _delay_ms(4);

}

void i2c_tx(char d)
{

  // SDA is LOW, SCL is HIGH (inherited from i2c-start command)

  // bring clock low to start data transfer

  i2cpin(SCL,LOW);
  
  char x;
  for(x=8; x; x--) 
  { 
    if(d&0x80) 
    {
      i2cpin(SDA,HIGH);  
    } else {
      i2cpin(SDA,LOW);  
    }

    i2cpin(SCL,HIGH);                  // strobe clock
    while (bit_is_clear(PINB,SCL)) {}  // wait until SCL has gone high (pick up on clock stretching)
    i2cpin(SCL,LOW);

    d <<= 1;                 // shift byte

  }

  i2cpin(SDA,HIGH);         // first bring SDA into listening mode

  // read acknowledge bit if any

  // an addressed i2c slave will have already set SDA low
  _delay_us(5);             // give the bus some time to possibly bring the pin high
  i2cpin(SCL,HIGH);         // start the clock pulse

  _delay_us(5);             // this should be 10us or so

  if (bit_is_set(PINB,SDA))
  {
    got_ack=0;
  } else {
    got_ack=1;
  }

  i2cpin(SDA,LOW);         // we now bring SDA low 
  _delay_us(10);          // give the bus some time?    ----- may go 

  i2cpin(SCL,LOW);          // end the clock pulse

  // slave detects SCL is low and will release SDA 


  _delay_ms(3);    // this should be 10us or so

}


void i2c_lcd_text(char *StrData)
{

  int p = 0;
  int q = strlen(StrData);
  int temp = 0;

  for (p = 0; p < q; p++)
  {
    temp = StrData[p];
    i2c_tx(temp);
  }
}


void i2c_lcd_number(int number)
{
  int digits;
  
  if (number == 0)
  {
    digits = 0;
    int array[0];
    array[0]=0;
    i2c_tx(array[0]+48);
  
  } else {

    // determine the number of digits
    digits = (int) (log(number)/log(10))+1;
    
    // split up the number's digits into an array
    int i = digits -1;
    int array[digits];
    while (number > 0)
    { 
      array[i--] = number % 10;
      number /= 10;
    }

    // send array over i2c
    for (i =0; i <= digits-1; i++)
    {
      i2c_tx(array[i]+48);
    }

  }
}

void i2c_lcd_binary(int number)
{
  // sends one byte in binary to the lcd
  
  int bit;
  int testbit;

  for (bit=7 ; bit >= 0; bit--)
  {
    testbit = number & (1 << bit);
    if (testbit)
    {
      i2c_tx(49);
    } else {
      i2c_tx(48);
    }

  }
}

void i2c_lcd_hex(uint8_t x)
{

  // sends one byte in hex (without 0x prefix) to the lcd

  char nibble;

  /* extract high nibble */
  nibble = x & 0xf0;        // (resets lower nibble to all zero's)
  nibble = x >> 4;          // (moves high nibble to lower nibble)

  // output the high nibble 
  if(nibble < 10) 
  {
    i2c_tx(48 + nibble);
  } else {
    i2c_tx(65 + nibble - 10);
  }

  /* do the same on the low nibble */
  nibble = x & 0x0f;

  if(nibble < 10) 
  {
    i2c_tx(48 + nibble);
  } else {
    i2c_tx(65 + nibble - 10);
  }

}


void delay_10us()
{
   unsigned long x=20;
   //    20 = 10us?
   //  2000 = 1ms?
   // 10000 = 5ms
   
   while (x > 0)
   {
     asm volatile("nop"::);
     x--;
   }

}


void delay_500us()
{
   unsigned long x=2000;
   //  2000 = 1ms?
   // 10000 = 5ms
   
   while (x > 0)
   {
     asm volatile("nop"::);
     x--;
   }
   


}

void trigger(int Pin)
{
  int x;

  digitalWrite(Pin, LOW);  
  pinMode(Pin, OUTPUT);    

  x=12; while (x > 0) { asm volatile("nop"::);  x--; }  // 10us delay

  digitalWrite(Pin, HIGH);
  pinMode(Pin, INPUT);
  
  x=6; while (x > 0) { asm volatile("nop"::);  x--; }  // 10us delay

  pinMode(Pin, OUTPUT);    
  digitalWrite(Pin, LOW);  

  x=12; while (x > 0) { asm volatile("nop"::);  x--; }  // 10us delay

  digitalWrite(Pin, HIGH);
  pinMode(Pin, INPUT);
  
  x=6; while (x > 0) { asm volatile("nop"::);  x--; }  // 10us delay

}

// -- OneWire functions --

inline __attribute__((gnu_inline)) void therm_delay(uint16_t delay)
{
  while(delay--) asm volatile("nop");
}

uint8_t owReset(int Pin)
{
  uint8_t i;

  digitalWrite(Pin, HIGH);
  pinMode(Pin, INPUT);

  digitalWrite(Pin, LOW);  
  pinMode(Pin, OUTPUT);    // bring low for 480us

  _delay_us(480);
 
  // Release line and wait for 60us
  digitalWrite(Pin, HIGH);
  pinMode(Pin, INPUT);
  _delay_us(60);

  // store line value and wait for completion of 480us period
  i=digitalRead(Pin);

  _delay_us(420);
  
  // return measured line value
  return i;

}

uint8_t therm_reset()
{
  uint8_t i;

  // pull line low and wait for 480us
  THERM_LOW();
  THERM_OUTPUT_MODE();
  therm_delay(480);
 
  // Release line and wait for 60us
  THERM_INPUT_MODE();
  therm_delay(60);

  // store line value and wait for completion of 480us period
  i=(THERM_PIN & (1<<THERM_DQ));
  therm_delay(420);
  
  // return measured line value
  return i;

}



void therm_write_bit(uint8_t bit)
{
  // pull line low for 1us
  THERM_LOW();
  THERM_OUTPUT_MODE();
  therm_delay(1);
  
  // if we want to write 1, release the line (if not will keep low)
  if (bit) THERM_INPUT_MODE();  

  // wait for 60us and release the line
  therm_delay(60);
  THERM_INPUT_MODE();

}

uint8_t therm_read_bit(void)
{
  uint8_t bit=0;

  // pull line low for 1us
  THERM_LOW();
  THERM_OUTPUT_MODE();
  therm_delay(1);

  // release line and wait for 14us
  THERM_INPUT_MODE();
  therm_delay(14);

  // read line value
  if (THERM_PIN&(1<<THERM_DQ)) bit=1;

  // wait for 45us to end and return read value
  therm_delay(45);

  return bit;

}

uint8_t therm_read_byte(void)
{

  uint8_t i=8, n=0;
 
  while(i--)
  {
    // shift one position right and store read value
    n>>=1;
    n|=(therm_read_bit()<<7);
  }

  return n;
}

void therm_write_byte(uint8_t byte)
{
  uint8_t i=8;

  while(i--)
  {
    // write actual bit and shift one position right to make the next bit ready
    therm_write_bit(byte&1);
    byte>>=1;
  }

}



unsigned char therm_rom_search(unsigned char diff, unsigned char *id)
{
  // scans a one-wire bus for all attached DS18x20 sensors

  unsigned char i, j, next_diff;
  unsigned char b;

  if( therm_reset() )
    return PRESENCE_ERR;                       // error, no device found
  therm_write_byte( THERM_CMD_SEARCHROM );     // ROM search command
  next_diff = LAST_DEVICE;                     // unchanged on last device
  i = 8 * 8;                        // 8 bytes
  do
  {
    j = 8;                          // 8 bits
    do
    {
      b = therm_read_bit();      // read bit

      // read complement bit
      if( therm_read_bit() ) 
      {         
        if( b )                     // 11
          return DATA_ERR;          // data error
      } else {
        if( !b )
        {
          // 00 = 2 devices
          if( diff > i || ((*id & 1) && diff != i) )
          {
            b = 1;                  // now 1
            next_diff = i;          // next pass 0
          }
        }
      }
      therm_write_bit( b );         // write bit
      *id >>= 1;
      if( b )                       // store bit
        *id |= 0x80;
      i--;
    } while( --j );

    id++;                           // next byte
  } while( i );

  return next_diff;                 // to continue search

} 




void start_meas( void )
{
  if( PINB & 1<< THERM_DQ )
  {
    therm_write_byte(THERM_CMD_CONVERTTEMP);    // read scratchpad

    PORTB |= 1<< THERM_DQ;
    DDRB |= 1<< THERM_DQ;			// parasite power on

  } else {

    strcpy_P(buffer, (char*)pgm_read_word(&(lcdmsg[00]))); 

    i2c_start();
    i2c_tx(116);           // device id (58)
    i2c_tx(12);            // clear screen
    i2c_lcd_text(buffer);  // 'Short circuit!'
    i2c_stop();      
          
    _delay_ms(1000);
  }
}


int scanbus(void)
{

  // outputs the number of found devices and stores all roms in an array
  unsigned char id[8], diff;
  unsigned char i,x;
  unsigned char device_counter=0;

  // clear array

  for (x=1; x <= 20; x++)
  {
    romcodes[x][0] = 0;
  }


  for( diff = SEARCH_FIRST; diff != LAST_DEVICE; )
  {
    diff = therm_rom_search( diff, id );

    if( diff == PRESENCE_ERR )
    {
      strcpy_P(buffer, (char*)pgm_read_word(&(lcdmsg[01]))); 

      i2c_start();
      i2c_tx(116);             // device id (58)
      i2c_tx(12);              // clear screen
      i2c_lcd_text(buffer);    // 'No sensors found'
      i2c_stop();                
      _delay_ms(1000);
      break;
    }
    if( diff == DATA_ERR )
    {
      strcpy_P(buffer, (char*)pgm_read_word(&(lcdmsg[02]))); 

      i2c_start();
      i2c_tx(116);             // device id (58)
      i2c_tx(12);              // clear screen
      i2c_lcd_text(buffer);    // 'Bus Error'
      _delay_ms(1000);
      i2c_stop();                
      break;
    }

    if( id[0] == 0x28 || id[0] == 0x10 )
    {
      // valid temperature sensor found

      // store romcode in array
      for( i = 0; i < 8; i++ )
      {

        romcodes[device_counter][i]=id[i];
        //i2c_lcd_hex(id[i]);

      }

      device_counter=device_counter+1;

    }
  }

  return device_counter;
}

void therm_romtemp(uint8_t device)
{
  // address a specific sensor and retrieve its temperature

  uint8_t i;

  therm_reset();
  therm_write_byte(THERM_CMD_MATCHROM);
  for (i=0 ; i < 8; i++)
  {
    therm_write_byte(romcodes[device][i]);
  }
  therm_write_byte(THERM_CMD_CONVERTTEMP);

  // wait until conversion is complete
  while(!therm_read_bit());



  therm_reset();
  therm_write_byte(THERM_CMD_MATCHROM);
  for (i=0 ; i < 8; i++)
  {
    therm_write_byte(romcodes[device][i]);
  }
  therm_write_byte(THERM_CMD_RSCRATCHPAD);

  // store the replies in the 'scratchpad' variable
  for (i=0 ; i < 2; i++)
  {
    scratchpad[i]=therm_read_byte();
  }
  therm_reset();

}


void showdevs()
{
  // outputs all device romcodes stored in the array

  unsigned char device=0;
  unsigned char rombyte=0;
  //uint8_t digit;
  uint16_t decimal;
 
  while (1)
  {

    if (romcodes[device][0] != 0)
    {

      // valid romcode found in array 

      // target the sensor and read its temperature
      therm_romtemp(device);   // this fills the 'scratchpad' variable

      strcpy_P(buffer, (char*)pgm_read_word(&(lcdmsg[03]))); 

      i2c_start();
      i2c_tx(116);               // device id (58)
      i2c_tx(12);                // clear screen
      i2c_lcd_text(buffer);      // 'Device '
      i2c_lcd_number(device+1);
      i2c_tx(10);                // position
      i2c_tx(2);                 // row 2
      i2c_tx(1);                 // col 1

      for (rombyte=0; rombyte < 8; rombyte++)
      {    
        i2c_lcd_hex(romcodes[device][rombyte]);            // display the romcode as hex on an LCD
      } 


      // store temperature integer digits and decimal digits

      //digit=scratchpad[0]>>4;
      //digit|=(scratchpad[1]&0x7)<<4;
      // store decimal digits
      //decimal=scratchpad[0]&0xf;
      //decimal*=THERM_DECIMAL_STEPS_12BIT;


      // display the 'scratchpad' variable fields

      // test if temperature is .0 or .5
      decimal = scratchpad[0] & (1 << 0);  // decimals is in bit 0, store value
      // move temperature bits on position to the right
      scratchpad[0]>>=1;

      i2c_tx(10);                    // position
      i2c_tx(1);                     // row 1
      i2c_tx(11);                    // col 11
      i2c_lcd_number(scratchpad[0]); //
      i2c_tx(46);                    // '.'

      if (decimal == 0)
      {
        i2c_tx(48);   // .0
      } else {
        i2c_tx(53);   // .5
      }

      i2c_tx(32);                    // ' '
      i2c_tx(67);                    // 'C'
      i2c_stop(); 

   
      _delay_ms(1000);

      device=device+1;

    } else {

      break;

    }

  } 

}



int main()
{

  uint8_t  ack;
  int devs;

  _delay_ms(1000);
  _delay_ms(1000);

  while (1)
  {

    ack = owReset(THERM_DQ);

    if (ack == 1)
    {

      strcpy_P(buffer, (char*)pgm_read_word(&(lcdmsg[01]))); 

      i2c_start();
      i2c_tx(116);            // device id (58)
      i2c_tx(12);             // clear screen
      i2c_lcd_text(buffer);   // 'No sensors found'
      i2c_stop();                
      _delay_ms(1000);

    } else {

      // sensor activity detected 

      while (1)
      {

        devs=scanbus();  // detect and store device rom's in an array

        if (devs > 0)
        {

          strcpy_P(buffer, (char*)pgm_read_word(&(lcdmsg[04]))); 

          i2c_start();
          i2c_tx(116);             // device id (58)
          i2c_tx(12);              // clear screen
          i2c_lcd_text(buffer);    // 'Devices: '
          i2c_lcd_number(devs);    // 
          i2c_stop();                
          _delay_ms(1000);
          _delay_ms(1000);
 
          showdevs();

        }

        _delay_ms(1000);
        _delay_ms(1000);

      }

    }

  }

  /*

           DS18S20

           byte 1                 byte 2
           -------------------------------
           76543210             76543210
           00101010             00000000
           IIIIDDDD             SSSSSIII 
           3210                      654 

           I = Celsius integer digits (signed)          000 0010 = 2
           D = Number of decimal steps (x multiplier)   1010     = 10 steps x 0.0625
           S = Signedness (0=positive, 1=negative)      00000    = +
   

           DS1820

           LSB (1st 7 bits are the temperature)
           and the last bit represents .0 or .5 degrees

           Example:   '87654321'
                       TTTTTTTH
           
             T = Temperature
             H = halves H=0 (.0), H=1 (.5)

           MSB (is 0 when positive, 1 when negative) 

  */

  return 0;

}

