/* Signal generator demonstration using a AD9850, encoder and 4x16 LCD

 Version: 0.03
 Last modified: 27 Sep 2014


 Changelog:
 v0.03
   - frequency is now put on output pin as soon as welcome screen disappears


 Set version number in 'show_welcome()' function below

 http://qrpme.blogspot.com.au/2013/03/bought-one-of-these-little-ad9850-dds.html

 Encoder connections:
 Middle pin to ground, outside two pins are connected to pins D2 and D3, so interrupts can be used

 AD9850 connections:
 8: CLK,  9: FQUP, 10: DATA, 11: RESET

 LCD connections:  (RW is tied to GND)
 7: RS, 6: EN, 5: D4, 4: D5, 3: D6, 2: D7


---
  AD9850 library
  
  Download the AH_AD9850 library (.h and .cpp files) from http://arduino.alhin.de/index.php?n=7
  
  then place the files in C:\program files\Arduino\hardware\arduino\avr\libraries\AH_AD9850\

  you can then include the library in your project as:
  
     #include <AH_AD9850.h>
---

*/

#include <LiquidCrystal.h>
#include <AH_AD9850.h>
#include <EEPROM.h>

LiquidCrystal lcd(7, 6, 5, 4, 3, 2);
AH_AD9850 AD9850(8, 9, 10, 11);

int ENC_LEFT=1;
int ENC_RIGHT=0;

int ENC_PWR_OUT=13;
int ENC_PWR_IN=12;

volatile uint8_t pinValues[2] = {0,0};
volatile unsigned long f = 14000000;
unsigned long step_size=1000000;
int nr_pos=9;
int prev_pos=9;
volatile int pos_mode=0;  // 0=freq 1=pos

unsigned int btn_down=0;
int freq_stored=0;

// values for storing the frequency in eeprom
//      14,000,000 MHz
//       | | | | | 
//       | | | | | 
//       | | | | | 
//       | | | | | 
//       | | | | | 
//    0-40 | | | +---------  0-99
//  addr:0 | | |           addr:4
//         | | +------   0-9
//         | |        addr:3
//         | +---  0-99
//       0-9     addr:2
//    addr:1

int mil_addr=0;
unsigned long mil_val=14;     // 0 - 40
int h_thou_addr=1;
unsigned long h_thou_val=0;   // 0 - 9
int thou_addr=2;
unsigned long thou_val=0;     // 0 - 99
int hun_addr=3;
int hun_val=0;                // 0 - 9
int ones_addr=4;
int ones_val=0;               // 0 - 99


void setup()
{
  lcd.begin(16, 4);
  AD9850.reset();

  // Configure encoder rotary pins as input, turn on pullup resistors
  pinMode(ENC_LEFT, INPUT);
  digitalWrite(ENC_LEFT, HIGH);
  pinMode(ENC_RIGHT, INPUT);
  digitalWrite(ENC_RIGHT, HIGH);

  attachInterrupt(0, knob_turned, CHANGE);
  attachInterrupt(1, knob_turned, CHANGE);
  
  // Configure encoder button press pins
  pinMode(ENC_PWR_OUT,OUTPUT);
  digitalWrite(ENC_PWR_OUT,LOW); // set to GND 
  
  pinMode(ENC_PWR_IN,INPUT);
  digitalWrite(ENC_PWR_IN,HIGH); // pullup so we can detect lows
  
  // display welcome message and show firmware version
  show_welcome();
  
}

void rotary_encoder_change(uint8_t changedPin, uint8_t value)
{
  pinValues[changedPin] = value;
  // only increment for each 'click' of the dial - when both pins have gone back to 0
  
  
  if (value == 0 && pinValues[0] == pinValues[1]) 
  {
    
    // button is in rest
    
    if (pos_mode==0)
    {
    
     
      
      // frequency change
      
      f += ((changedPin) ? step_size : -step_size);
  
      // only allow frequencies between 0Hz - 40MHz
      if ((f >= 0) && (f <= 40000000))    
      {
          AD9850.set_frequency(0,0,f);
      } else {
          // revert change to f
          f += ((changedPin) ? -step_size : +step_size);
      }
      
  
          
    } else {
     
      // decimal position change
      
      // set the new position value
      prev_pos = nr_pos;
      nr_pos += ((changedPin) ? -1 : 1);
      
      if ((nr_pos >= 1) && (nr_pos <= 9))
      {
        // within range
        
        if (nr_pos > prev_pos)
        {
          // coming from the right
          if (nr_pos==4) { nr_pos=5; }
          if (nr_pos==8) { nr_pos=9; } 
        } else {
          // coming from the left
          if (nr_pos==4) { nr_pos=3; }
          if (nr_pos==8) { nr_pos=7; } 
        }
        
        // update step value
        switch (nr_pos) 
        {
          case 1:   
            step_size=1; 
            break;
          case 2:    
            step_size=10; 
            break;        
          case 3:
            step_size=100; 
            break;
          case 5:    
            step_size=1000; 
            break;
          case 6:    
            step_size=10000;
            break;
          case 7:    
            step_size=100000;
            break;
          case 9:
            step_size=1000000;
            break;
          default:            
            step_size=1000000;
        }
        
      } else {
        
        // going outside of range, revert back
        nr_pos = prev_pos;
        
      }
      
    }
    
  }
  
}

void knob_turned()
{
  delay(1);
  int pin0 = digitalRead(ENC_LEFT);
  int pin1 = digitalRead(ENC_RIGHT);
  if (pin0 != pinValues[0]) {
      rotary_encoder_change(0, pin0);
  }
  else if (pin1 != pinValues[1]) {    
      rotary_encoder_change(1, pin1);
  }
}

void update_display()
{
  
  if (pos_mode==0)
  {
    
    // mode 0 = up/down frequency 
    
    char buf[17];
    int mhz = f/1000000;
    int khz = (f - mhz * 1000000) / 1000;
    int hz = f - mhz * 1000000 - khz * 1000;
    sprintf (buf, "   %d,%03d,%03dHz", mhz, khz, hz);  // formatting of the value
  
    if (f<10000000)
    {
      lcd.setCursor(1,0); // shift value one pos to the right
    } else {
      lcd.setCursor(0,0); // value occupies two digits for 10's of MHz.
    }
  
    lcd.print(buf);
    sprintf (buf, "%ld", f);  // print frequency as per the formatting
    lcd.setCursor(-4,2);
    lcd.print(buf);
  
  } else {
   
      // mode 1 = move cursor 
      
        
        if (nr_pos != prev_pos)
        {
          lcd.setCursor(13-prev_pos,1); // move cursor 
          lcd.print(' '); // wipe out previous cursor position
        }
        lcd.setCursor(13-nr_pos,1); // move cursor
        lcd.print('-');  // wipe out previous cursor position
      
     
  }
}

void loop()
{

  // store frequency if button has been pressed more than 2 seconds 
  if (btn_down > 200)
  {
    store_freq(f);
    pos_mode=0;
  }
  
  update_display(); // show freq or cursor
  
  btn_down=0;
  
  while (!(digitalRead(ENC_PWR_IN)))
  {
   
    
      // toggle position mode when button is pressed
      if (pos_mode==0)
      {
          pos_mode=1; // decimal position mode
        
      } else {
            
        pos_mode=0; // frequency mode 
        lcd.setCursor(13-nr_pos,1);
        lcd.print(' ');
   
      }
  
    
   
  
    
    // count here how long the button is pressed  
    while (!(digitalRead(ENC_PWR_IN))) 
    {
      // count the number of 10's of ms the button has been held down for
      btn_down = btn_down + 1;
      delay(10);
    }
 
    delay(100);
   
  }
 
}


void get_freq()
{
   // retrieve frequency stored in eeprom and set to active frequency
  
   mil_val    = EEPROM.read(mil_addr);
   h_thou_val = EEPROM.read(h_thou_addr);
   thou_val   = EEPROM.read(thou_addr);
   hun_val    = EEPROM.read(hun_addr);
   ones_val   = EEPROM.read(ones_addr);
  
   f = (mil_val*1000000) + (h_thou_val*100000) + (thou_val*1000) + (hun_val*100) + ones_val;
  
}


void store_freq(unsigned long freq)
{
   // store current frequency in on-board eeprom
  
   // break up value in sub variables
  
   mil_val = int(freq/1000000);
   freq = freq - (mil_val*1000000);
   h_thou_val = int(freq/100000);
   freq = freq - (h_thou_val*100000);
   thou_val = int(freq/1000);
   freq = freq - (thou_val*1000);
   hun_val   = int(freq/100);
   freq = freq - (hun_val*100);
   ones_val  = freq;
  
   EEPROM.write(mil_addr, mil_val);
   EEPROM.write(h_thou_addr, h_thou_val);
   EEPROM.write(thou_addr, thou_val);
   EEPROM.write(hun_addr, hun_val);
   EEPROM.write(ones_addr, ones_val);
   
   lcd.setCursor(0,1);
   lcd.print("Frequency stored");
   delay(1000);
   lcd.setCursor(0,1);
   lcd.print("                ");
}



void show_welcome()
{
  
   lcd.setCursor(0,0);
   lcd.print("  0 - 40MHz AWG");
   lcd.setCursor(0,1);
   lcd.print("    Ver: 0.03");

   delay(1000);
   delay(1000);

   lcd.setCursor(0,0);
   lcd.print("                ");
   lcd.setCursor(0,1);
   lcd.print("                ");
   lcd.setCursor(0,0);
   
   // get frequency from eeprom and use as active frequency
   get_freq();
   
   // set frequency on output pin
   AD9850.set_frequency(0,0,f);
}
