/* 
   I2C SLAVE - LCD display with 3 wires using 74HC595
   --------------------------------------------------

   Software version: 1.0 (compatible with hardware version 1.0)
   Author: M.Post
   Last modified: 20 Jan 2012


   DONE:

      - switch button now on PB4 (CLK pin of LCD) so it 
        doesn't interfere with LCD while pressing the button


 
   TODO:
     
     - eeprom value for I2C id may need to be set explicitly for brand new mcu's.
        I wrote a simple function that checks if the value is 255 but I'm not sure
        if this is adequate for new chips. If your display hangs at startup it may 
        just be that you need to write an eeprom value first:
           - change code to write an I2C value (e.g. 58)
           - program the chip
           - change the code back to only write a code if it's 255
           - program the chip

     - add buttons to send I2C commands

     special instructions:
       - clear one row
       - turn on backlight  (Need 4.2V. Resistor or PWM on pin ???)
            pins that can do pwm: 
             PB0 output pin for PWM mode timer function
             PB1 output pin for PWM mode timer function 
             PB3 inverted output pin for PWM mode timer function
             PB4 output pin for PWM mode timer function

http://www.nongnu.org/avr-libc/user-manual/group__demo__project.html


   getting started in asm
   http://attiny13.lars.stonerocket.co.uk/
   http://www.nongnu.org/avr-libc/user-manual/inline_asm.html

   asm volatile("nop\n\t"
                "nop\n\t"

   ::);


   C bitwise operations: 
   http://stackoverflow.com/questions/47981/how-do-you-set-clear-and-toggle-a-single-bit-in-c


   I2C examples from:
   http://www.robot-electronics.co.uk/acatalog/I2C_Tutorial.html
 
   PIC12F code by Rajendra Bhatt:
   http://www.electronics-lab.com/projects/mcu/015/index.html
   Ported to ATtiny85 by M.Post
 

                             ATtiny85
               
                             +--- ---+
       backlight  (RST) PB5 -|1 |_| 8|- VCC
                    EN  PB3 -|2     7|- PB2 SCL    (SCK)          
    setup switch    CLK PB4 -|3     6|- PB1 SDA    (MISO)         
                        GND -|4     5|- PB0 Dout   (MOSI)
                             +-------+


   Pololu programming cable pinout (top view)

                          +---+
    ----------------  gnd |o o| rst
    ribbon cable     mosi |o o| sck
    -----red--------  vdd |o o| miso
                          +---+

    NOTE: The T85 must be externally powered with 5V (4.5V does NOT work!)
          A t85 driving a standard 2x16 LCD without backlit LED on draws
          about 8.5mA so with 100mA to spare on a USB port you could consider
          grabbing 5V from the Pololu programmer. The 5V pin on the 
          programmer is labeled: 'VBUS (+5V)'

       

      LCD Connector: GND VCC EN DOUT CLK

      LCD Pinout
      ----------
      15 Led Power Anode
      16 Led Power Cathode
      1 Vss (GND)
      2 Vdd (+5V)
      3 VO
      4 RS ----------- to pin  4 on 74HC595 (Q4)
      5 R/W
      6 E ------------ to 'EN' pin on t85
      7 DB0
      8 DB1
      9 DB2
      10 DB3
      11 DB4 --------- to pin 15 on 74HC595 (Q0)
      12 DB5 --------- to pin  1 on 74HC595 (Q1)
      13 DB6 --------- to pin  2 on 74HC595 (Q2)
      14 DB7 --------- to pin  3 on 74HC595 (Q3)


      74HC595
      -------
      pinout as per above, plus:
       - pin 11 (SH_CP) and pin 12 (ST_CP) are tied together
       - above pins to to CLK pin on t85
       - pin 14 (DS) to Dout pin on t85
       - pin 8 and 13 to GND
       - pin 16 and 10 to +5V


      Converter board pinout
      ----------------------
      
          +----------------------------------------+
          |                                        |
          |     +---------------------+            |
          |     |                     |    o +5V   |
          |     |   LCD Module        |    o SCL   |
          |  p  |                     |    o SDA   |
          |     +---------------------+    o GND   |
          |      r1 74HC595 t85  btn  isp          |
          |                                        |
          +----------------------------------------+

     




 */

// clock is defined on gcc command line using 'DF_CPU=8000000UL'
// you may therefore need to program the lower fuse byte to 0xE1 (disable clkdiv8)

#include <avr/io.h>
#include <util/delay.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>
#include <avr/eeprom.h>

// pins to be used on ATtiny85
#define STR  PB3    // Define Strobe/Enable pin  (doubles as setup switch pin)
#define Dout PB0    // Define Data-Out pin 
#define CLK  PB4    // Define Clock pin  
#define SCL  PB2    // Define I2C Clock pin  
#define SDA  PB1    // Define I2C Data pin  
#define BACKLIGH  PB5  // Define LED Backlight pin


#define HIGH   1
#define LOW    0
#define INPUT  1 
#define OUTPUT 0


int device_id;                 // this slave's device id (defaults to decimal 58)
int id_match;                  // high when master addresses us
int start=0;                   // repeated start indicator
unsigned int i2c_timeout=65535;       // 4185000=5sec   // when to give up listening for more bits  (837,000 is about 1 second @ 8MHz)

void digitalWrite(int pin, int state)
{
  if (state == HIGH)
  { 
    (PORTB |= (1 << pin)); 
  } else {
    (PORTB &= ~(1 << pin));
  }
} 

int digitalRead(int pin)
{ 
  if (bit_is_set(PINB,pin))
  {
    return 1; 
  } else { 
    return 0; 
  }
} 



void pinMode(int pin,int state)
{ 
  if (state == OUTPUT)
  {
    (DDRB |= (1 << pin));  // set pin to output
  } else { 
    (DDRB &= ~(1 << pin)); // set pin to input
  }
} 

void i2cpin(int line, int state)
{

  if (state == 0)
  {
    pinMode(line,OUTPUT);
    digitalWrite(line,LOW); 
  } else {
    pinMode(line,INPUT);
  }

}



void Write_LCD_Nibble(unsigned short N, int RS)
{

  digitalWrite(STR,0);

  // ****** Write RS *********
  digitalWrite(CLK,0);
  digitalWrite(Dout,RS);  // Data_Pin
  digitalWrite(CLK,1);
  digitalWrite(CLK,0);


  // Shift in 4 bits
  int t = 0;
  int Flag = 0;
  int Mask = 8;

  for (t=0; t<4; t++)
  {
    Flag = N & Mask;
    if(Flag==0)
    {
      digitalWrite(Dout,0);  // Data_Pin     
    } else {
      digitalWrite(Dout,1);  // Data_Pin
    }
 
    digitalWrite(CLK,1);
    digitalWrite(CLK,0);

    Mask = Mask >> 1;
  }

  // One more clock because SC and ST clks are tied
  digitalWrite(CLK,1);
  digitalWrite(CLK,0);
  digitalWrite(Dout,0);  // Data_Pin
  digitalWrite(STR,1);   // Enable_Pin  (to LCD)
  digitalWrite(STR,0);   // Enable_Pin  (to LCD)

}


void Write_LCD_Data(unsigned short D)
{
  int RS = 1; // It is Data, not command
  int Low_Nibble = D & 15;
  int High_Nibble = D/16;
  Write_LCD_Nibble(High_Nibble,RS);
  Write_LCD_Nibble(Low_Nibble,RS);
}

void Write_LCD_Cmd(unsigned short C)
{

  int RS = 0; // It is command, not data
  int Low_Nibble = C & 15;
  int High_Nibble = C/16;
  Write_LCD_Nibble(High_Nibble,RS);
  Write_LCD_Nibble(Low_Nibble,RS);
}



void initialise_LCD() 
{

  pinMode(CLK,0);
  pinMode(STR,0);
  pinMode(Dout,0);


  _delay_ms(100); 

  Write_LCD_Cmd(0x03); // function set
  _delay_ms(50);

  Write_LCD_Cmd(0x03);
  _delay_ms(50);

  Write_LCD_Cmd(0x03);
  _delay_ms(50);

  Write_LCD_Cmd(0x02);
  _delay_ms(50);

  Write_LCD_Cmd(0x2C); // Display ON, No cursors
  _delay_ms(50);

  Write_LCD_Cmd(0x06); // Entry mode- Auto-increment, No Display shifting
  _delay_ms(50);

  Write_LCD_Cmd(0x0E);
  _delay_ms(50);

  Write_LCD_Cmd(0x01);
  _delay_ms(50);

  Write_LCD_Cmd(0x80);
  _delay_ms(50);


}




void Position_LCD(unsigned short x, unsigned short y)
{
  int temp = 127 + y;

  if (x == 2) 
  {
    temp = temp + 64;
  }

  Write_LCD_Cmd(temp);
}
 

void Clear_LCD()
{
   Write_LCD_Cmd(0x01);  // Clear LCD
   _delay_ms(5);

}

void Write_LCD_Text(char *StrData)
{

  int p = 0;
  int q = strlen(StrData);
  int temp = 0;

  for (p = 0; p < q; p++)
  {
    temp = StrData[p];
    Write_LCD_Data(temp);
  }
}



void Write_LCD_Number(char number)
{
  // writes a number's digits on the display

  int digits;

  if (number == 0)
  {
    digits = 0;
    int array[0];
    array[0]=0;
    Write_LCD_Data(array[0]+48);

  } else {
 
    // determine the number of digits 
    int digits = (int)(log(number)/log(10))+1;

    // split up the number to an array
    int i = digits - 1;
    int array[digits];
    while (number > 0)
    {
      array[i--] = number % 10;
      number /= 10;  
    }

    // display array on LCD
    for(i = 0; i <= digits-1; i++)
    {
      Write_LCD_Data(array[i]+48);
    }

  }

}


void get_or_set_i2c_device_id()
{
  // retrieve device id from on-system eeprom
  device_id=eeprom_read_byte(0);
  _delay_ms(10);

 
  if (device_id == 255)
  {
    device_id=58;                 // initial value if eeprom is empty
    eeprom_write_byte(0,device_id);
    _delay_ms(10);
  }
}



void setup_t85()
{


  initialise_LCD();

  i2cpin(SDA,HIGH);  
  i2cpin(SCL,HIGH);  

}


void wait_for_start()
{
  // exits loop whenever a start is detected and clock goes low

  while (1)
  {
    
    while (bit_is_clear(PINB,SDA)) {} // wait for SDA to go high 
    while (bit_is_set(PINB,SDA)) {} // wait for SDA to go low
    if (bit_is_set(PINB,SCL))
    {
      // i2c start sequence detected, exit loop    
      break;
    } 

  }

  while (bit_is_set(PINB,SCL)) {} // exit when SCL goes low

}


unsigned char i2c_rxbyte()
{


  unsigned char rxbyte=0;
  int bitcount=8;
  unsigned int counter;
  int gotbit;

  // stop clock stretching: we now release SCL 
  (DDRB &= ~(1 << SCL)); // set pin to input


  // (we received a valid start sequence and) SCL and SDA are now low

  while (bitcount > 0)
  {  
    gotbit=0;
    counter=0;

    // in any case, we release SCL
    //(DDRB &= ~(1 << SCL)); // set pin to input


    // wait for SCL to go high 

    while (counter < i2c_timeout) 
    {
      if (bit_is_set(PINB,SCL))
      {
        if (bit_is_set(PINB,SDA))
        {
          // -- detected bit '1' --
          
          rxbyte |= 1 << 0 ; // sets bit 0 of rxbyte

        } else {

          // -- detected bit '0' --

        }
        break;       
      }  
      counter++;
    }  

   
    // if (gotbit == 0)
    if (counter == i2c_timeout)
    {
      // timeout reached! (SCL never went high) 
      // stop listening for further bits/bytes

      rxbyte=0;
      id_match=0;
      break;
    }

    // still here? means we received a valid SCL-high (could be a bit or a stop)

    // shift rxbyte and continue listening
    if (bitcount > 1)
    {
      rxbyte = rxbyte << 1; // shift all bits in rxbyte one position to the left
    }

    // scl is high 
    // sda is high
  
    counter=0;  // int(i2c_timeout*2);  // because this loop is twice as slow?
    gotbit=0;  
    // wait for SCL to go low again

    // (only listen for i2c-STOP if SDA is low)

    if (bit_is_set(PINB,SDA))
    {
      // SDA is high, we'll never receive an i2c-STOP now
      // just listen for SCL to go low

      while (counter < i2c_timeout)
      {
        if (bit_is_clear(PINB,SCL))
        {
          // SCL is now low
          gotbit=1;
          break; 
        }
      }

    } else {

      // SDA is low, we could see SCL go low (received bit) or SDA go high (received STOP)

      // do this only for the 8th bit (MSB comes first)
      if (bitcount==8)
      {

        while (counter < i2c_timeout)
        {
          if (bit_is_clear(PINB,SCL))
          {
            // SCL is now low
            gotbit=1;
            break; 
          }

          if (bit_is_set(PINB,SDA))
          {
            // SDA went high, we received an i2c-STOP sequence
            gotbit=0;
            id_match=0;
            
            break; 
          }
          counter++;
        }

      } else {

        // listen for SCL to go low

        while (counter < i2c_timeout)
        {
          if (bit_is_clear(PINB,SCL))
          {
            // SCL is now low
            gotbit=1;
            break; 
          }
        }

      }

    }



    if (gotbit == 0)
    {
      // timeout reached or i2c-STOP received. Stop listening for further bits/bytes
     
      if (counter==0)
      {
        // first wait for SCL to go low again
        while (bit_is_set(PINB,SCL)) {}


      }


      rxbyte=0;
      id_match=0;
      break;

    } else {
    
      // received a valid bit or byte
      

    }

    bitcount--;

  }

  return rxbyte;
}





void wait_for_devmatch()
{

  unsigned char rxbyte;
  int dev_rx_id = device_id;
  dev_rx_id <<= 1;                // shift byte  

  while (1)
  {
    wait_for_start();            // loop until we detect an i2c start sequence
    rxbyte = i2c_rxbyte();       // listen for a byte being transmitted

    if (rxbyte == dev_rx_id)
    {
      break;
    }

  } 
 
  // don't forget to send an acknowledge after this command has finished

}

void send_i2c_ack()
{

  unsigned int counter=0;
  int gotbit=0;

  // we bring SDA low 
  (DDRB |= (1 << SDA));     // set pin to output
  (PORTB &= ~(1 << SDA));   // write a '0'

  // wait until master brings SCL high 
  while (counter < i2c_timeout)
  {
    if (bit_is_set(PINB,SCL))
    {
      gotbit=1; 
      break;
    } 
    counter++;
  }
   
  if (gotbit == 0)
  {

    // timeout reached! (SCL never went high) 
    // stop listening for further bits/bytes
    id_match=0;

  } else {

    // SCL is now high (set by master)

    // the master will now detect that we have set SDA low

    // wait until master brings SCL low
    gotbit=0;
    while (counter < i2c_timeout)
    {
      if (bit_is_clear(PINB,SCL))
      {
        gotbit=1; 
        break;
      } 
      counter++;
    }
   
    if (gotbit == 0)
    {

      // timeout reached! (SCL never went low) 
      // stop listening for further bits/bytes
      id_match=0;

    } 

    // if gotbit=1, SCL is now low
    // the master has now ended the clock (SCL is low) and has taken over SDA again 

    // SDA state unknown
    // SCL is low


  }



  // begin clock streching: bring SCL low until we are ready
  (DDRB |= (1 << SCL));     // set pin to output
  (PORTB &= ~(1 << SCL));   // write a '0'

  // in any case, we release SDA
  (DDRB &= ~(1 << SDA)); // set pin to input

}



void setup_menu()
{
  // CLK is the switch pin

  int new_dev_id=device_id;
  int counter=0;
  int setup_exit=0;
  char dev_id[]="Device ID: ";

  Clear_LCD();
  Position_LCD(1,1);
  Write_LCD_Text(dev_id);

  while (1)
  {

    // display current value of the device id

    Position_LCD(1,12);
    Write_LCD_Number(new_dev_id);

    pinMode(CLK,1);    // change pinmode so we can listen for button presses
    // pinMode(STR,1);
    // pinMode(Dout,1);


    while (bit_is_clear(PINB, CLK)) {}  // wait until we receive a button press (switch pin)
    _delay_ms(50);

    // see if we hold the button down for a long time to indicate an exit
    counter=0; 
    setup_exit=0;
    while (bit_is_set(PINB, CLK))   // set switch pin here
    {
      _delay_ms(1);
      if (counter >= 3000)
      {
        setup_exit=1;
        while (bit_is_set(PINB, CLK)) {}  // wait until button is release (switch pin)
        break;
      }
      counter++;
    } 
    pinMode(CLK,0);  // set pin to output to enable LCD writing


    if (setup_exit==0)
    {
      new_dev_id++;
      if (new_dev_id > 127)
      { 
        new_dev_id=1;
        Clear_LCD();
        Position_LCD(1,1);
        Write_LCD_Text(dev_id);
      }
    } else {

      // write new device id to eeprom
      eeprom_write_byte(0,new_dev_id);
      device_id=new_dev_id;

      initialise_LCD();
       
      char updated[]="* UPDATED * ";
      Position_LCD(1,1);
      Write_LCD_Text(dev_id);
      Position_LCD(1,12);
      Write_LCD_Number(new_dev_id);
      Position_LCD(2,1);
      Write_LCD_Text(updated);

      _delay_ms(1000); 
      _delay_ms(1000); 
 
      break;

    }
  
  }

}

void enter_setup_check()
{
  // check if Switch is pressed down on startup for longer than 1 second
  pinMode(CLK,1);   // set CLK to input  (switch pin)

  if (bit_is_set(PINB, CLK)) 
  {
    // push button is pressed
   
    while (bit_is_set(PINB,CLK))  { };   // wait until button is released

    pinMode(CLK,0);
    pinMode(STR,0);
    pinMode(Dout,0);

    initialise_LCD();

    char setup_welcome[]="Entering setup..";
    Write_LCD_Text(setup_welcome);

    _delay_ms(1000);
    _delay_ms(1000);
    _delay_ms(1000);

    setup_menu();

  }

}




int main()
{

  get_or_set_i2c_device_id();

  enter_setup_check();

  setup_t85();

  int rxbyte=0;

  char Message1[] = "I2C LCD Module";
  char Message2[] = "Listening..";

  Position_LCD(1,1);
  Write_LCD_Text(Message1);

  Position_LCD(2,1);
  Write_LCD_Text(Message2);

  _delay_ms(1000);

  Clear_LCD();



  int LCD_Row, LCD_Col; 


  while (1)
  {
    

    id_match=0;
    wait_for_devmatch();

    id_match=1;
    send_i2c_ack();       // send acknowledge 


    // listen for incoming data..
    while (id_match==1)
    { 
      // listen for data packets 

      rxbyte = i2c_rxbyte();

      if (rxbyte == 0)
      {

        // no more valid data received, exit this loop
        break;

      } else {

        // valid byte received

        send_i2c_ack();       // send acknowledge 


        // only send valid ASCII characters (32-126) to the LCD
        if ((rxbyte >= 32) && (rxbyte <= 126))
        {
 
          // regular character

          Write_LCD_Data(rxbyte);

        } else {

          // special character

          // clear screen
          if (rxbyte == 12)
          { 

            Clear_LCD(); 

          }

          // positioning of the cursor on the LCD
          if (rxbyte == 10)
          {

            // first byte is LCD row (1-2)
            LCD_Row=i2c_rxbyte();   // Read data and send acknowledge
            // send acknowledge
            send_i2c_ack();       // send acknowledge 
     

            // second byte is LCD column (1-16)
            LCD_Col=i2c_rxbyte();   // Read data and send acknowledge
            // send acknowledge
            send_i2c_ack();       // send acknowledge 


            // position the cursor
            Position_LCD(LCD_Row,LCD_Col);


          }

        } // end if disecting what kind of data byte we received


        // received byte has now been processed

      
      } // end-if when receiving one byte


    } // while-loop for receiving bytes addressed to us


  } // main listening loop

  return 0;
}
