/*

   AVR Fuse Resetter  
   -----------------

   Last updated: 26 October 2011


                        ATtiny84
               
                        +--   --+
        (Vdd) --- Vdd --|1 |_|14|-- GND --- (GND)
    I2C - SDA --- PB0 --|2    13|-- PA0 --- 
    I2C - SCL --- PB1 --|3    12|-- PA1 --- 
        (RST) --- PB3 --|4    11|-- PA2 ---
              --- PB2 --|5    10|-- PA3 ---
              --- PA7 --|6     9|-- PA4 --- (SCK)
       (MOSI) --- PA6 --|7     8|-- PA5 --- (MISO)
                        +-------+



   Pololu programming cable pinout (top view)

                          +---+
    ----------------  gnd |o o| rst
    ribbon cable     mosi |o o| sck
    -----red--------  vdd |o o| miso
                          +---+

    NOTE: The T84 must be externally powered with 5V (4.5V does NOT work!)
          A t85 driving a standard 2x16 LCD without backlit LED on draws
          about 8.5mA so with 100mA to spare on a USB port you could consider
          grabbing 5V from the Pololu programmer. The 5V pin on the 
          programmer is labeled: 'VBUS (+5V)'


byte (not in winavr/gcc), solution: 
typedef unsigned char byte; //create byte type

PORTx controls the voltage levels 0V or 5V (0 or 1)
DDRx controls direction, Output or Input (0 or 1)
PINx reads which voltage levels are low or high


DONE:

- I2C LCD coded added and is now working properly
- added I2C device id to i2c_start command (left shift)
- added I2C code stretch wait loop timeout


TODO:

- Add ATtiny fuse reset code
- I2C code: determine line status after sending a '1' as the last byte (either from byte or ack?)




*/

#include <avr/io.h>
#include <util/delay.h>
#include <shiftOut.h>
#include <math.h>    // for 'log' function (in 'i2c_lcd_number()' to determine the number of digits)
#include <string.h>  // needed? 

#define HIGH      1
#define LOW       0
#define INPUT     1
#define OUTPUT    0

// I2C pin definitions, all on PORTB
#define SDA  PB0    // Define Data pin  
#define SCL  PB1    // Define Clock pin  

// pushbutton pin
#define BUTTON PB2  // pushbutton pin

// Programmer pin definitions, all on PORTA
//                       ard  t84  t85  Description
#define INSTOUT PA0  //  10    8    6   Connect to Serial Instruction Input (SII)
#define DATAOUT PA1  //   9    7    5   Connect to Serial Data Input (SDI)
#define CLKOUT  PA2  //  12    2    2   Connect to Serial Clock Input (SCI)
#define RST     PA3  //  13    4    1   Output to level shifter for !RESET from transistor
#define DATAIN  PA4  //  11    9    7   Connect to Serial Data Output (SDO)
#define VCC     PA7  //   8    1    8   Connect to VCC

// Desired fuse configuration
#define  HFUSE  0xDF   // Defaults for ATtiny25/45/85
#define  LFUSE  0x62   


typedef unsigned char byte; //create byte type


int inByte = 0;         // incoming serial byte Computer (change to button press)
int inData = 0;         // incoming serial byte AVR

char blankline[]="                ";


int got_ack;
byte LCD_I2C_DEVICE_ID=58;
int I2C_STRETCH_TIMEOUT_MS=100;


// --- DIGITAL PIN OPERATIONS ---
// --- DIGITAL PIN OPERATIONS ---
// --- DIGITAL PIN OPERATIONS ---

void pinMode(int pin,int state)
{ 
  if (state == OUTPUT)
  {
    (DDRA |= (1 << pin)); 
  } else { 
    (DDRA &= ~(1 << pin)); 
  }
} 

void digitalWrite(int pin, int state)
{
  // make sure port is set to output
  DDRA |= (1 << pin);

  if (state == HIGH)
  { 
    (PORTA |= (1 << pin)); 
  } else {
    (PORTA &= ~(1 << pin));
  }
} 

int digitalRead(int pin)
{ 
  if (bit_is_set(PINA,pin))
  {
    return 1; 
  } else { 
    return 0; 
  }
} 

// --- I2C Functions ---
// --- I2C Functions ---
// --- I2C Functions ---

void i2cpin(int line, int state)
{
  // set the I2C SDA or SCL PIN

  if (state == 0)
  {
    // set the pin low

    DDRB |= (1 << line);     // set line to output
    PORTB &= ~(1 << line);   // set it to zero

  } else {

    // set the pin high (let external pull-up resistor bring it high)

    DDRB &= ~(1 << line);  // set line to input
   
  }

  //_delay_ms(500);   // use this as a debug tool
 

}

void i2c_delay()
{
  // default delay in between I2C commands
  _delay_ms(10);   // seems to hover around 2-3ms
}


void i2c_tx(char d)
{

  // SDA is LOW, SCL is HIGH (inherited from i2c-start command)

  // bring clock low to start data transfer

  i2cpin(SCL,LOW);

     
  
  char x;
  //static int b;
  for(x=8; x; x--) 
  { 
    if(d&0x80) 
    {
      i2cpin(SDA,HIGH);  
    } else {
      i2cpin(SDA,LOW);  
    }

    i2cpin(SCL,HIGH);                  // strobe clock


    // allow for I2C clock stretching (SCL can be held low by slave)
    // see I2C_STRETCH_TIMEOUT_MS

    int stretch_counter;
    
    for (stretch_counter=0; stretch_counter < I2C_STRETCH_TIMEOUT_MS; stretch_counter++)
    {
      // allow for some cycles for SCL to go high  

      if (bit_is_clear(PINB,SCL))
      {
        // SCL is held low by slave
        // we'll just wait a little 

        _delay_ms(1);

      } else {

        // SCL has risen, ready to move on
        stretch_counter = I2C_STRETCH_TIMEOUT_MS;
      }


    }

    //_delay_us(1);                      // 1-3 us  
    i2cpin(SCL,LOW);

    d <<= 1;                 // shift byte

  }

  i2cpin(SDA,HIGH);         // first bring SDA into listening mode

  //_delay_ms(1);             // delay no longer needed, this used to be 1.6ms
  //_delay_us(600);

  // read acknowledge bit if any

  // an addressed i2c slave will have already set SDA low
  _delay_us(5);             // give the bus some time to possibly bring the pin high
  i2cpin(SCL,HIGH);         // start the clock pulse

  _delay_us(5);             // this should be 10us or so

  if (bit_is_set(PINB,SDA))
  {
    got_ack=0;
  } else {
    got_ack=1;
  }

  i2cpin(SDA,LOW);         // we now bring SDA low 
  _delay_us(10);          // give the bus some time?    ----- may go 

  i2cpin(SCL,LOW);          // end the clock pulse

  // slave detects SCL is low and will release SDA 


  _delay_ms(3);    // this should be 10us or so


  // return b;

}



void i2c_start(char i2c_id)
{
  // SDA goes from high to low while SCL is high

  i2cpin(SDA,HIGH); 
  _delay_us(10);    // give bus some time to bring SDA high
  i2cpin(SCL,HIGH);

  _delay_us(10);    

  i2cpin(SDA,LOW);
  _delay_us(10);    


  // send the I2C Device ID

  // send as Write (last bit is 0)

  
  i2c_tx(i2c_id<<1);



}

void i2c_stop()
{
  // SDA goes from low to high while SCL is high

  i2cpin(SDA,LOW);
  _delay_us(20);    // give bus some time to bring SDA high
  i2cpin(SCL,LOW); 

  _delay_ms(6);

  i2cpin(SCL,HIGH);
  _delay_ms(6);

  i2cpin(SDA,HIGH);
  _delay_ms(4);

}


void i2c_lcd_text(char *StrData)
{

  int p = 0;
  int q = strlen(StrData);
  int temp = 0;

  for (p = 0; p < q; p++)
  {
    temp = StrData[p];
    i2c_tx(temp);
  }
}

void i2c_lcd_number(int number)
{
  int digits;

  if (number == 0)
  {
    digits = 0;
    int array[0];
    array[0]=0;
    i2c_tx(array[0]+48);

  } else {

    // determine the number of digits
    digits = (int) (log(number)/log(10))+1;

    // split up the number's digits into an array
    int i = digits -1;
    int array[digits];
    while (number > 0)
    {
      array[i--] = number % 10;
      number /= 10;
    }

    // send array over i2c
    for (i =0; i <= digits-1; i++)
    {
      i2c_tx(array[i]+48);
    }

  }
}

void i2c_lcd_hex(uint8_t x)
{

  // sends one byte in hex (without 0x prefix) to the lcd

  char nibble;

  /* extract high nibble */
  nibble = x & 0xf0;        // (resets lower nibble to all zero's)
  nibble = x >> 4;          // (moves high nibble to lower nibble)

  // output the high nibble
  if(nibble < 10)
  {
    i2c_tx(48 + nibble);
  } else {
    i2c_tx(65 + nibble - 10);
  }

  /* do the same on the low nibble */
  nibble = x & 0x0f;

  if(nibble < 10)
  {
    i2c_tx(48 + nibble);
  } else {
    i2c_tx(65 + nibble - 10);
  }

}




// --- FUSE RESET CODE ---
// --- FUSE RESET CODE ---
// --- FUSE RESET CODE ---




int shiftOut2(uint8_t dataPin, uint8_t dataPin1, uint8_t clockPin, uint8_t bitOrder, byte val, byte val1)
{
	int i;
        int inBits = 0;
        //Wait until DATAIN goes high
        while (!digitalRead(DATAIN));
        
        //Start bit
        digitalWrite(DATAOUT, LOW);
        digitalWrite(INSTOUT, LOW);
        digitalWrite(clockPin, HIGH);
  	digitalWrite(clockPin, LOW);
        
	for (i = 0; i < 8; i++)  {
                
		if (bitOrder == LSBFIRST) {
			digitalWrite(dataPin, !!(val & (1 << i)));
                        digitalWrite(dataPin1, !!(val1 & (1 << i)));
                }
		else {
			digitalWrite(dataPin, !!(val & (1 << (7 - i))));
                        digitalWrite(dataPin1, !!(val1 & (1 << (7 - i))));
                }
                inBits <<=1;
                inBits |= digitalRead(DATAIN);
                digitalWrite(clockPin, HIGH);
		digitalWrite(clockPin, LOW);
                
	}

        
        //End bits
        digitalWrite(DATAOUT, LOW);
        digitalWrite(INSTOUT, LOW);
        digitalWrite(clockPin, HIGH);
        digitalWrite(clockPin, LOW);
        digitalWrite(clockPin, HIGH);
        digitalWrite(clockPin, LOW);
        
        return inBits;
}



void enableProgramming()
{

    // Initialize pins to enter programming mode

    pinMode(DATAIN, OUTPUT);  //Temporary
    digitalWrite(DATAOUT, LOW);
    digitalWrite(INSTOUT, LOW);
    digitalWrite(DATAIN, LOW);
    digitalWrite(RST, HIGH);  // Level shifter is inverting, this shuts off 12V
    
    // Enter High-voltage Serial programming mode
    digitalWrite(VCC, HIGH);  // Apply VCC to start programming process
    _delay_us(20);
    digitalWrite(RST, LOW);   //Turn on 12v
    _delay_us(10);
    pinMode(DATAIN, INPUT);   //Release DATAIN
    _delay_us(300);

    //Now in programming mode
}


void disableProgramming()
{

    //Exiting programming Mode
    digitalWrite(CLKOUT, LOW);
    digitalWrite(VCC, LOW);
    digitalWrite(RST, HIGH);   //Turn off 12v
}


void readFuses()
{
    char reading[]="Reading..";
    char heading[]=" LOW  HIGH   EXT";
    char nulx[]="0x";

    enableProgramming();
    

    i2c_start(LCD_I2C_DEVICE_ID);
    i2c_tx(12);
    i2c_lcd_text(reading);
    i2c_stop();   
             
    _delay_ms(500);

    i2c_start(LCD_I2C_DEVICE_ID);
    i2c_tx(12);  // clear lcd
    i2c_lcd_text(heading);
    i2c_tx(10); // jump to row 2
    i2c_tx(2);
    i2c_tx(1);


     //Read lfuse
    shiftOut2(DATAOUT, INSTOUT, CLKOUT, MSBFIRST, 0x04, 0x4C);
    shiftOut2(DATAOUT, INSTOUT, CLKOUT, MSBFIRST, 0x00, 0x68);
    inData = shiftOut2(DATAOUT, INSTOUT, CLKOUT, MSBFIRST, 0x00, 0x6C);
    i2c_lcd_text(nulx);
    i2c_lcd_hex(inData);
    i2c_tx(32);
    i2c_tx(32);   
    
    //Read hfuse
    shiftOut2(DATAOUT, INSTOUT, CLKOUT, MSBFIRST, 0x04, 0x4C);
    shiftOut2(DATAOUT, INSTOUT, CLKOUT, MSBFIRST, 0x00, 0x7A);
    inData = shiftOut2(DATAOUT, INSTOUT, CLKOUT, MSBFIRST, 0x00, 0x7E);
    i2c_lcd_text(nulx);
    i2c_lcd_hex(inData);
    i2c_tx(32);
    i2c_tx(32);    
    
    //Read efuse
    shiftOut2(DATAOUT, INSTOUT, CLKOUT, MSBFIRST, 0x04, 0x4C);
    shiftOut2(DATAOUT, INSTOUT, CLKOUT, MSBFIRST, 0x00, 0x6A);
    inData = shiftOut2(DATAOUT, INSTOUT, CLKOUT, MSBFIRST, 0x00, 0x6E);
    i2c_lcd_text(nulx);
    i2c_lcd_hex(inData);

    i2c_stop();  

    disableProgramming();


}


void writeFuses()
{

    char writing[]="Writing..";


    enableProgramming();

    i2c_start(LCD_I2C_DEVICE_ID);
    i2c_tx(12);
    i2c_lcd_text(writing);
    i2c_stop();   
             
    _delay_ms(500);

    //Write hfuse
    shiftOut2(DATAOUT, INSTOUT, CLKOUT, MSBFIRST, 0x40, 0x4C);
    shiftOut2(DATAOUT, INSTOUT, CLKOUT, MSBFIRST, HFUSE, 0x2C);
    shiftOut2(DATAOUT, INSTOUT, CLKOUT, MSBFIRST, 0x00, 0x74);
    shiftOut2(DATAOUT, INSTOUT, CLKOUT, MSBFIRST, 0x00, 0x7C);

    
    //Write lfuse
    shiftOut2(DATAOUT, INSTOUT, CLKOUT, MSBFIRST, 0x40, 0x4C);
    shiftOut2(DATAOUT, INSTOUT, CLKOUT, MSBFIRST, LFUSE, 0x2C);
    shiftOut2(DATAOUT, INSTOUT, CLKOUT, MSBFIRST, 0x00, 0x64);
    shiftOut2(DATAOUT, INSTOUT, CLKOUT, MSBFIRST, 0x00, 0x6C);

    disableProgramming();
    

}


// --- BASIC FUNCTIONS ---
// --- BASIC FUNCTIONS ---
// --- BASIC FUNCTIONS ---



void delay_secs(int secs)
{

  while (secs > 0)
  {
    _delay_ms(1000);
    secs --;
  }

}

// --- MAIN PROGRAM ---
// --- MAIN PROGRAM ---
// --- MAIN PROGRAM ---



int main()
{

  // -- set up lines for I2C --
  pinMode(SDA,INPUT);
  pinMode(SCL,INPUT);
  pinMode(BUTTON,INPUT);

  PORTB = 0;    // disable internal pullup resistors

  PORTB &= ~(1 << SDA);  // set SDA LOW (makes line output)
  PORTB &= ~(1 << SCL);  // set SCL LOW (makes line output)



  // -- set up control lines for HV parallel programming --
  pinMode(VCC, OUTPUT);
  pinMode(RST, OUTPUT);
  pinMode(DATAOUT, OUTPUT);
  pinMode(INSTOUT, OUTPUT);
  pinMode(CLKOUT, OUTPUT);
  pinMode(DATAIN, OUTPUT);  // configured as input when in programming mode
  
  // Initialize output pins as needed
  digitalWrite(RST, HIGH);  // Level shifter is inverting, this shuts off 12V





  char message1[]="AVR FuseResetter";
  char message2[]="short press=read";
  char message3[]="long press=write";
  
  _delay_ms(1000);



  while (1)
  {

    // show welcome message
    i2c_start(LCD_I2C_DEVICE_ID);
    i2c_tx(12);          // clear display
    i2c_lcd_text(message1);
    i2c_tx(10);
    i2c_tx(2);
    i2c_tx(1);
    i2c_lcd_text(message2);
    i2c_stop();                

    delay_secs(3);

    i2c_start(LCD_I2C_DEVICE_ID);
    i2c_tx(10);
    i2c_tx(2);
    i2c_tx(1);
    i2c_lcd_text(message3);
    i2c_stop();                

    delay_secs(3);

    i2c_start(LCD_I2C_DEVICE_ID);
    i2c_tx(10);
    i2c_tx(2);
    i2c_tx(1);
    i2c_lcd_text(blankline);
    i2c_stop();                
              


    while (bit_is_clear(PINB,BUTTON)) {  }  // wait until keypress is detected     

    i2c_start(LCD_I2C_DEVICE_ID);
    i2c_tx(12);  // clear lcd
    i2c_stop(); 

    _delay_ms(100);

    int btcounter=0;

    while (bit_is_set(PINB,BUTTON))
    {
      btcounter++;
      _delay_ms(10);
     
    }

    // button has been released


    i2c_start(LCD_I2C_DEVICE_ID);
    i2c_tx(12);  // clear lcd
    i2c_lcd_number(btcounter);
    i2c_stop();                


    _delay_ms(100);

    if (btcounter < 200)
    {
  
      readFuses();  // was 0xE1 0xDF 0xFF


    } else {

      
      writeFuses();

  
      i2c_start(LCD_I2C_DEVICE_ID);
      i2c_tx(10);
      i2c_tx(2);
      i2c_tx(1);
      i2c_lcd_text(blankline);
      i2c_tx(10);
      i2c_tx(2);
      i2c_tx(1);
      i2c_stop();                

      readFuses(); 


    }

    delay_secs(5);
   
  
  

  }

  return 0;

}

