/* 
   RS232 temperature sensor board
   ------------------------------

   Changelog:
   v2 -- released 25 April 2014
    - added support for DS18B20
   
   v1 -- initial release
  

   clock is defined on gcc command line using 'DF_CPU=16000000UL'

                           ATtiny85
               
                           +--- ---+
                      PB5 -|1 |_| 8|- VCC 
    RX (PC) ------->  PB3 -|2     7|- PB2  -----> TX (PC)
    DQ (DS1820) ----  PB4 -|3     6|- PB1           
                      GND -|4     5|- PB0  
                           +-------+
   Code by: M. Post
   Last modified: 25 April 2014

   NOTE: The AVR ISP protocol is incompatible with the DQ pin of the temperature sensor.
         Make sure the DQ pin is on a non-ISP pin.
   
   
 */


#include <avr/io.h>
#include <util/delay.h>
#include <string.h>
#include <math.h>
#include <avr/pgmspace.h>


#define Bit_Delay   104  // sets RS-232 speed

// RS232 pin definitions
#define Sin  PB2    // Define Serial-In pin   (PC-TX)
#define Sout PB3    // Define Serial-Out pin  (PC-RX)

#define HIGH   1
#define LOW    0
#define INPUT  1 
#define OUTPUT 0
#define MARK   0 
#define SPACE  1

#define THERM_PORT   PORTB
#define THERM_DDR    DDRB
#define THERM_PIN    PINB
#define THERM_DQ     PB4        // Define DS1820 DQ pin

#define THERM_INPUT_MODE()      THERM_DDR&=~(1<<THERM_DQ)
#define THERM_OUTPUT_MODE()     THERM_DDR|=(1<<THERM_DQ)
#define THERM_LOW()             THERM_PORT&=~(1<<THERM_DQ)
#define THERM_HIGH()            THERM_PORT|=(1<<THERM_DQ)

#define THERM_CMD_CONVERTTEMP    0x44
#define THERM_CMD_RSCRATCHPAD    0xbe
#define THERM_CMD_WSCRATCHPAD    0x4e
#define THERM_CMD_CPYSCRATCHPAD  0x48
#define THERM_CMD_RECEEPROM      0xb8
#define THERM_CMD_RPWRSUPPLY     0xb4
#define THERM_CMD_SEARCHROM      0xf0
#define THERM_CMD_READROM        0x33
#define THERM_CMD_MATCHROM       0x55
#define THERM_CMD_SKIPROM        0xcc
#define THERM_CMD_ALARMSEARCH    0xec

#define THERM_DECIMAL_STEPS_12BIT   625    // 0.0625

#define PRESENCE_ERR 0xFF
#define DATA_ERR     0xFF
#define LAST_DEVICE  0x00
#define SEARCH_FIRST 0xFF

#define CLEAR(s) memset(&(s), 0, sizeof(s))     // to clear an array

int      got_ack;
uint8_t  romcodes[10][9];  // store romcode of .. sensors max  (REMOVE! use on-the-fly detection)
uint8_t  scratchpad[9];    // stores a scratchpad

char crlf[] = "\r\n";
char ready[] = "Ok\r\n";
char buserror[] = "#";
char romword[] = "ROM ";
char nosensors[] = "No sensors!\r\n";
char welcome[] = "USB/Serial Temperature Board 2014-04-26 Marcel Post\r\n";


void pinMode(int pin,int state)
{ 
  if (state == OUTPUT)
  {
    (DDRB |= (1 << pin)); 
  } else { 
    (DDRB &= ~(1 << pin)); 
  }
} 

void digitalWrite(int pin, int state)
{
  if (state == HIGH)
  { 
    (PORTB |= (1 << pin)); 
  } else {
    (PORTB &= ~(1 << pin));
  }
} 


int digitalRead(int pin)
{ 
  if (bit_is_set(PINB,pin))
  {
    return 1; 
  } else { 
    return 0; 
  }
} 


// --  begin RS232 functions --

void initialise_Serial(long int baud)
{

  pinMode(Sout,OUTPUT);
  pinMode(Sin,INPUT);

  // work out mark and space lengths based on clock speed

  /*
        bps     bit length (us)
   --------------------------
        2400       416.7
        4800       208.3
        9600       104.2
       19200        52.1
       38400        26.0
       57600        17.4
      115200         8.7

  */
 
 
}

void Write_Serial_Data(unsigned short D)
{

  // start bit is high, stop bit is low
  // character bits: 0=high, 1=low
  // mark=0 (LOW, 0 Volts)
  // space=1 (HIGH, 5 Volts)

  (PORTB &= ~(1 << Sout));       // MARK (bring low)

  _delay_us(Bit_Delay);              

  int t;
  for (t=0 ; t <=7 ; t++)
  {
   if ((D) & (1<<(t)))
   {
      (PORTB |= (1 << Sout));        // SPACE (bring high)
    } else {
      (PORTB &= ~(1 << Sout));       // MARK (bring low)
    }  
    _delay_us(Bit_Delay); 
  }

  (PORTB |= (1 << Sout));            // SPACE (bring high)
  _delay_us(Bit_Delay);           

}

void Write_Serial_Text(char *StrData)
{

  int p = 0;
  int q = strlen(StrData);
  char temp = 0;

  for (p = 0; p < q; p++)
  {
    temp = StrData[p];
    Write_Serial_Data(temp);
  }
  
}

void Write_Serial_Number(int number)
{
  int digits;
  
  if (number == 0)
  {
    digits = 0;
    int array[0];
    array[0]=0;
    Write_Serial_Data(array[0]+48);
  
  } else {

    // determine the number of digits
    digits = (int) (log(number)/log(10))+1;
    
    // split up the number's digits into an array
    int i = digits -1;
    int array[digits];
    while (number > 0)
    { 
      array[i--] = number % 10;
      number /= 10;
    }

    // send array over i2c
    for (i =0; i <= digits-1; i++)
    {
      Write_Serial_Data(array[i]+48);
    }

  }
}

void Write_Serial_Hex(uint8_t x)
{

  // sends one byte in hex (without 0x prefix) 

  char nibble;

  /* extract high nibble */
  nibble = x & 0xf0;        // (resets lower nibble to all zero's)
  nibble = x >> 4;          // (moves high nibble to lower nibble)

  // output the high nibble 
  if(nibble < 10) 
  {
    Write_Serial_Data(48 + nibble);
  } else {
    Write_Serial_Data(65 + nibble - 10);
  }

  /* do the same on the low nibble */
  nibble = x & 0x0f;

  if(nibble < 10) 
  {
    Write_Serial_Data(48 + nibble);
  } else {
    Write_Serial_Data(65 + nibble - 10);
  }

}

// --  end RS232 functions --




// -- Begin OneWire functions --

inline __attribute__((gnu_inline)) void therm_delay(uint16_t delay)
{
  while(delay--) asm volatile("nop");
}

uint8_t owReset(int Pin)
{
  uint8_t i;

  digitalWrite(Pin, HIGH);
  pinMode(Pin, INPUT);

  digitalWrite(Pin, LOW);  
  pinMode(Pin, OUTPUT);    // bring low for 480us

  _delay_us(480);
 
  // Release line and wait for 60us
  digitalWrite(Pin, HIGH);
  pinMode(Pin, INPUT);
  _delay_us(60);

  // store line value and wait for completion of 480us period
  i=digitalRead(Pin);

  _delay_us(420);
  
  // return measured line value
  return i;

}

uint8_t therm_reset()
{
  uint8_t i;

  // pull line low and wait for 480us
  THERM_LOW();
  THERM_OUTPUT_MODE();
  therm_delay(480);
 
  // Release line and wait for 60us
  THERM_INPUT_MODE();
  therm_delay(60);

  // store line value and wait for completion of 480us period
  i=(THERM_PIN & (1<<THERM_DQ));
  therm_delay(420);
  
  // return measured line value
  return i;

}



void therm_write_bit(uint8_t bit)
{
  // pull line low for 1us
  THERM_LOW();
  THERM_OUTPUT_MODE();
  therm_delay(1);
  
  // if we want to write 1, release the line (if not will keep low)
  if (bit) THERM_INPUT_MODE();  

  // wait for 60us and release the line
  therm_delay(60);
  THERM_INPUT_MODE();

}

uint8_t therm_read_bit(void)
{
  uint8_t bit=0;

  // pull line low for 1us
  THERM_LOW();
  THERM_OUTPUT_MODE();
  therm_delay(1);

  // release line and wait for 14us
  THERM_INPUT_MODE();
  therm_delay(14);

  // read line value
  if (THERM_PIN&(1<<THERM_DQ)) bit=1;

  // wait for 45us to end and return read value
  therm_delay(45);

  return bit;

}

uint8_t therm_read_byte(void)
{

  uint8_t i=8, n=0;
 
  while(i--)
  {
    // shift one position right and store read value
    n>>=1;
    n|=(therm_read_bit()<<7);
  }

  return n;
}

void therm_write_byte(uint8_t byte)
{
  uint8_t i=8;

  while(i--)
  {
    // write actual bit and shift one position right to make the next bit ready
    therm_write_bit(byte&1);
    byte>>=1;
  }

}

unsigned char therm_rom_search(unsigned char diff, unsigned char *id)
{
  // scans a one-wire bus for all attached DS18x20 sensors

  unsigned char i, j, next_diff;
  unsigned char b;

  if( therm_reset() )
    return PRESENCE_ERR;                       // error, no device found
  therm_write_byte( THERM_CMD_SEARCHROM );     // ROM search command
  next_diff = LAST_DEVICE;                     // unchanged on last device
  i = 8 * 8;                        // 8 bytes
  do
  {
    j = 8;                          // 8 bits
    do
    {
      b = therm_read_bit();      // read bit

      // read complement bit
      if( therm_read_bit() ) 
      {         
        if( b )                     // 11
          return DATA_ERR;          // data error
      } else {
        if( !b )
        {
          // 00 = 2 devices
          if( diff > i || ((*id & 1) && diff != i) )
          {
            b = 1;                  // now 1
            next_diff = i;          // next pass 0
          }
        }
      }
      therm_write_bit( b );         // write bit
      *id >>= 1;
      if( b )                       // store bit
        *id |= 0x80;
      i--;
    } while( --j );

    id++;                           // next byte
  } while( i );

  return next_diff;                 // to continue search

} 

void start_meas( void )
{
  char shortcircuit[] = "Short!\r\n";

  if( PINB & 1<< THERM_DQ )
  {
    therm_write_byte(THERM_CMD_CONVERTTEMP);    // read scratchpad

    PORTB |= 1<< THERM_DQ;
    DDRB |= 1<< THERM_DQ;			// parasite power on

  } else {

    Write_Serial_Text(shortcircuit); // short circuit!
    _delay_ms(1000); 

  }
}

int scanbus(void)
{

  // outputs the number of found devices and stores all roms in an array
  unsigned char id[8], diff;
  unsigned char i,x;
  unsigned char device_counter=0;


  // clear array

  for (x=1; x <= 20; x++)
  {
    romcodes[x][0] = 0;
  }


  for( diff = SEARCH_FIRST; diff != LAST_DEVICE; )
  {
    diff = therm_rom_search( diff, id );

    if( diff == PRESENCE_ERR )
    {
      Write_Serial_Text(crlf); 
      Write_Serial_Text(nosensors);  // No Sensors found
      _delay_ms(1000);
      break;
    }
    if( diff == DATA_ERR )
    {
      //Write_Serial_Text(buserror);  // Bus Error
      _delay_ms(1000);
      break;
    }

    if( id[0] == 0x28 || id[0] == 0x10 )
    {
      // valid temperature sensor found

      // store romcode in array
      for( i = 0; i < 8; i++ )
      {
        romcodes[device_counter][i]=id[i];
      }

      device_counter=device_counter+1;

    }
  }

  return device_counter;
}

void therm_romtemp(uint8_t device)
{
  // address a specific sensor and retrieve its temperature

  uint8_t i;

  therm_reset();
  therm_write_byte(THERM_CMD_MATCHROM);
  for (i=0 ; i < 8; i++)
  {
    therm_write_byte(romcodes[device][i]);
  }
  therm_write_byte(THERM_CMD_CONVERTTEMP);

  // wait until conversion is complete
  while(!therm_read_bit());

  // send the eight bytes of the ROM code to the DS1820
  therm_reset();
  therm_write_byte(THERM_CMD_MATCHROM);
  for (i=0 ; i < 8; i++)
  {
    therm_write_byte(romcodes[device][i]);
  }
  therm_write_byte(THERM_CMD_RSCRATCHPAD);

  // store the replies in the 'scratchpad' variable
  // get eight values 
  for (i=0 ; i < 8; i++)
  {
    scratchpad[i]=therm_read_byte();
  }
  therm_reset();

}


void showdevs()
{
  // outputs all device romcodes stored in the array

  unsigned char device=0;
  unsigned char rombyte=0;
  
  while (1)
  {

    if (romcodes[device][0] != 0)
    {

      // valid romcode found in array 

      // target the sensor and read its temperature
      therm_romtemp(device);   // this fills the 'scratchpad' variable

      Write_Serial_Text(romword);     // ROM


      for (rombyte=0; rombyte < 8; rombyte++)
      {  
        Write_Serial_Data(48);    // 0
        Write_Serial_Data(120);   // x
        Write_Serial_Hex(romcodes[device][rombyte]);            // display the romcode as hex
        Write_Serial_Data(32);    // space

      } 

      // store temperature integer digits and decimal digits


      // display the 'scratchpad' variable fields

	  
	  int16_t decimals_big;
      uint16_t digits_s; 
      uint16_t fulltemp_s;
	  
	  uint8_t digits_b;
	  uint8_t fulltemp_b;
	  
	  uint8_t decimals;
 
      if (romcodes[device][0]==0x10)
      {
		// two-decimal (9bit) accuracy for DS1820 and DS18S20        
	  
	    /* 
		  DS18S20 Temperature Register Format
		   
		            Bit 7  Bit 6  Bit 5  Bit 4  Bit 3  Bit 2  Bit 1  Bit 0
		  LS Byte    2^6    2^5    2^4    2^3    2^2    2^1    2^0    2^-1
		  
		            Bit15  Bit14  Bit13  Bit12  Bit11  Bit10  Bit 9  Bit 8
		  MS Byte     S      S      S      S      S      S      S      S
		
		  S = Sign
		  
		*/
	  
        digits_s=scratchpad[0]>>1;     // get the whole-digits temperature 
        digits_s=digits_s*100;           // multiply by 100  (e.g. 21 -> 2100)
        digits_s=digits_s-25;            // subtract (0.25), here: 2100 - 25 = 2075

        // calculate the decimal digits
        // COUNT_REMAIN [6]
        // COUNT_PER_C  [7]
        decimals = ((scratchpad[7]*100)-(scratchpad[6]*100))/scratchpad[7];    // this will give us four decimals

        fulltemp_s = digits_s + decimals;

        digits_s = (int) fulltemp_s / 100;
        decimals = fulltemp_s - (digits_s * 100);

        // compensate for negative temperatures
        if ( scratchpad[1] > 0 ) 
        {
          digits_s = 127 - digits_s;
          if ( decimals > 0 )
          {
            decimals=100-decimals;
          } else {
            digits_s = digits_s + 1;
          }
          Write_Serial_Data(45);                  // '-'
        }
 

        Write_Serial_Number(digits_s); 
        Write_Serial_Data(46);                    // '.'
 
        if (decimals < 10)
        {
          Write_Serial_Number(0);   // leading zero for decimal values < 10
        }
        Write_Serial_Number(decimals); 
        
      }

      if (romcodes[device][0]==0x28)
	  {
	  
	    // two-decimal (default 12bit) accuracy for DS18B20 
		// (code added 25 April 2014)
	   
	   /* 
		  DS18B20 Temperature Register Format
		   
		            Bit 7  Bit 6  Bit 5  Bit 4  Bit 3  Bit 2  Bit 1  Bit 0
		  LS Byte    2^3    2^2    2^1    2^0    2^-1   2^-2   2^-3   2^-4
		  
		            Bit15  Bit14  Bit13  Bit12  Bit11  Bit10  Bit 9  Bit 8
		  MS Byte     S      S      S      S      S     2^6    2^5    2^4
		
		  S = Sign
		
		*/
		
		
		// sign (not yet written)		
		//Write_Serial_Data(45);                  // '-'
		
		
	    // digits and decimals
		
		// extract digits
		// right-shift LSB 4 places to remove the decimals
		fulltemp_b = scratchpad[0]>>4;
		
		// add bit 8,9 and 10 from the MSB
		digits_b = scratchpad[1]<<4;  // first, move MSB 4 places to the left	
		digits_b &= ~(1<<7);          // then clear bit 7 (as it holds the pos/neg sign)
		digits_b = digits_b + fulltemp_b; // lastly, add both together
		
		// extract decimals
		// clear bits 7,6,5 and four
		decimals = scratchpad[0];
		
		decimals &= ~(1<<7); // clear bit 7
		decimals &= ~(1<<6); // clear bit 6
		decimals &= ~(1<<5); // clear bit 5
		decimals &= ~(1<<4); // clear bit 4	
		
		// show sign for negative temperatures
		// test if 7th bit is set (0=temp is positive, 1=temp is negative)
        if ( scratchpad[1] & (1<<7) ) 
        {
			// temperature is negative
			 Write_Serial_Data(45);                    // '-'
			
			// with negative temperatures, the digits and decimals are counted from the top
			
			// for the decimals we subtract from 16
			decimals = 16 - decimals;
			
			if (decimals==16)
			{
				decimals=0;
				digits_b = digits_b - 1;				
			}
			
			// for the digits we subtract from 127
			digits_b = 127 - digits_b;
			
		}
		
		Write_Serial_Number(digits_b); 
        Write_Serial_Data(46);                    // '.'
		
		/* decimals now holds the multiplier of the minimum precision
	       since the default precision is 12-bits the minimum precision is 0.0625
	  	   decimals can potentially hold up to 16 (0 - 15).
		   we multiply 0.0625 * the value in decimals to get the correct decimals
		   
		   Example:
		   
		     digits = 14
		     14 x 0.0625 = 0.875
		     to avoid non-integer numbers in our calculations, we multiply by 10,000
		     so..
		     0.0625 x 10,000 = 625
		     14 x 625 = 8750
		     then we divide by 100 to truncate the last two decimals
		
		     so that..
		     1 x 625 =  625 >>2 =  6
		     2 x 625 = 1250 >>2 = 12
		     3 x 625 = 1875 >>2 = 18
		     4 x 625 = 2500 >>2 = 25
		     5 x 625 = 3125 >>2 = 31
		     6 x 625 = 3750 >>2 = 37
		     7 x 625 = 4375 >>2 = 43
		     8 x 625 = 5000 >>2 = 50
		     9 x 625 = 5625 >>2 = 56
		    10 x 625 = 6250 >>2 = 62
		    11 x 625 = 6875 >>2 = 68
		    12 x 625 = 7500 >>2 = 75
		    13 x 625 = 8125 >>2 = 81		   
		    14 x 625 = 8750 >>2 = 87
		    15 x 625 = 9375 >>2 = 93
		*/
		
		decimals_big = decimals * 625;
		decimals_big = decimals_big / 100;
		decimals = (int) decimals_big;
		
		// add a leading zero for decimal values below 10
		if (decimals < 10)
		{
			Write_Serial_Number(0);   // leading zero for decimal values < 10
        }
        Write_Serial_Number(decimals); 
		
	  }

      Write_Serial_Data(32);     // ' '
      Write_Serial_Data(67);     // 'C'
      Write_Serial_Data(32);     // ' '
      Write_Serial_Text(crlf);  // crlf
    
      device=device+1;

    } else {

      break;

    }

  } 

}

// -- end OneWire functions --



int main()
{
  initialise_Serial(9600);

  _delay_ms(1000);

  uint8_t  ack;
  int devs;

  while (1)
  {

    // wait until we detect incoming serial data; our cue to get query the sensors
    while (bit_is_set(PINB,Sin)) {}      

    _delay_ms(100);


    ack = owReset(THERM_DQ);

    if (ack == 1)
    {

      Write_Serial_Text(crlf); 
      Write_Serial_Text(nosensors);  // 'No sensors found'
      _delay_ms(1000);

    } else {

      // sensor activity detected 

      devs=scanbus();  // detect and store device rom's in an array

      if (devs > 0)
      {
        showdevs();
      }

    }


    Write_Serial_Text(welcome);  // welcome message
    Write_Serial_Text(ready);    // Ready
    Write_Serial_Text(crlf); 

    // _delay_ms(1000);

  }

  return 0;

}
